package cn.pconline.search.common;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import cn.pconline.search.common.log.SearchLogger;
import cn.pconline.search.common.tools.homonym.Homonyms;

/**
 * 搜索框架
 * 
 * @author zengjie
 * @since 2013-9-22
 * @see
 */
public class SearchFrame
{

    private volatile boolean inited = false;

    private static Logger logger = Logger.getLogger(SearchFrame.class);

    protected Configuration configuration;

    private SearchLogger searchLogger;

    private SolrSearcher searcher;

    private Homonyms homonyms;

    /**
     * 对框架通过指定的配置进行初始化
     * 
     * @param config
     */
    public void init(Configuration config)
    {

        if (!inited)
        {
            synchronized (this)
            {
                if (inited)
                {
                    return;
                }
                long start = System.currentTimeMillis();
                logger.debug("Init SearchFrame with config{" + config + "}");
                // 是否开启搜索日志记录
                if (config.getBooleanConfig("enableSearchLog", true))
                {
                    searchLogger = new SearchLogger(config);
                }
                searcher = new SolrSearcher(config);
                // 初始化同音词(如果配置中存在同音词典文件)
                String homonymsFiles = config.getConfig("homonymsFiles", null);
                if (StringUtils.isNotBlank(homonymsFiles))
                {
                    homonyms = new Homonyms(homonymsFiles);
                }
                logger.debug("SearchFrame init complete in "
                        + (System.currentTimeMillis() - start) + " ms");
                inited = true;
            }
        }
    }

    private SearchFrame()
    {
    }

    private void checkInit()
    {
        if (!inited)
        {
            throw new IllegalStateException(
                    "Frame has not been initialize or has been closed,pelease make sure has invoked #init# method");
        }
    }

    private static SearchFrame instance = new SearchFrame();

    /**
     * 获取单例，使用前确保调用了init方法
     * 
     * @return
     */
    public static SearchFrame get()
    {
        return instance;
    }

    /**
     * 获取搜索日志记录器
     * 
     * @return
     */
    public SearchLogger getLogger()
    {
        checkInit();
        return searchLogger;
    }

    /**
     * 获取对solr进行访问的Searcher
     * 
     * @return
     */
    public SolrSearcher getSearcher()
    {
        checkInit();
        return searcher;
    }

    /**
     * 获取当前框架上的同音词典
     * 
     * @return
     */
    public Homonyms getHomonyms()
    {
        checkInit();
        return homonyms;
    }

    /**
     * 获取同音词
     * 
     * @see SearchFrame#getHomonyms()
     * @param key
     * @return
     */
    public String getHomonynsWords(String key)
    {
        checkInit();
        if (homonyms == null)
        {
            return "";
        }
        return homonyms.getHomonyms(key);
    }

    /**
     * 关闭当前搜索框架对象,所有获取操作立即不可用
     */
    public void shutdown()
    {
        if (!inited)
        {
            return;
        }
        try
        {
            searcher.close();
            searcher = null;
            searchLogger.close();
            searchLogger = null;
            homonyms = null;
        }
        finally
        {
            inited = false;
        }
    }

    /**
     * 获取当前搜索框架是否可用
     * 
     * @return
     */
    public boolean isAvaliable()
    {
        return inited;
    }
}
