package cn.pconline.search.common;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpHost;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.NTCredentials;
import org.apache.http.client.HttpClient;
import org.apache.http.conn.params.ConnRoutePNames;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.impl.HttpSolrServer;
import org.apache.solr.client.solrj.request.QueryRequest;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;

import com.alibaba.fastjson.JSON;

import cn.pconline.search.common.ks.KSResult;
import cn.pconline.search.common.util.Intf;
import cn.pconline.search.common.util.URLHolder;

/**
 * solr搜索器
 * 
 * @author zengjie
 * @since 2013-9-16
 * @see
 */
public class SolrSearcher
{

    public static final String DEFAULT_SOLR_URL = "http://192.168.239.79:8983/solr/";

    private SolrServer server;

    private Configuration config;

    /**
     * constructor
     * 
     * @param solrBaseUrl
     * @param uniqueName
     *            用来缓存Searcher的全局唯一key
     */
    public SolrSearcher(String solrBaseUrl, Configuration config)
    {
        HttpSolrServer httpServer = new HttpSolrServer(solrBaseUrl);
        this.config = config;
        setupHttpServer(httpServer);
        this.server = httpServer;
    }

    public SolrSearcher(Configuration config)
    {
        this(config.getConfig("searchServer", DEFAULT_SOLR_URL), config);
    }

    /**
     * 执行搜索操作,返回solr的查询响应,默认使用select handler
     * 
     * @param index
     * @param q
     * @return
     */
    public QueryResponse search(String index, SolrQuery q)
    {
        return search(index, "select", q);
    }

    /**
     * 使用指定的handler搜索指定的index
     * 
     * @param index
     * @param handler
     * @param q
     * @return
     */
    public QueryResponse search(String index, String handler, SolrQuery q)
    {
        QueryRequest req = new QueryRequest(q);
        req.setPath("/" + index + "/" + handler);
        try
        {
            return req.process(server);
        }
        catch (SolrServerException e)
        {
            throw new SearchException(e);
        }
    }

    /**
     * 对旧快搜进行搜索并且返回结果
     * 
     * @param searchUrl
     * @return
     * @throws IOException
     */
    public KSResult searchKs(URLHolder searchUrl) throws IOException
    {
        searchUrl.getParams().addOrReplace("returnType", "json");
        String rsp = Intf.readHtml4Get(searchUrl.toString(), null, "gbk", 3000,
                2);
        if (StringUtils.isEmpty(rsp))
        {
            return new KSResult();
        }
        return JSON.parseObject(rsp.trim(), KSResult.class);
    }

    /**
     * 对HTTP Server的属性进行相关设置
     * 
     * @param server
     */
    protected void setupHttpServer(HttpSolrServer server)
    {
        server.setConnectionTimeout(config.getIntConfig("http.ConnectTimeout",
                5000));
        server.setSoTimeout(config.getIntConfig("http.SoTimeout", 5000));
        server.setMaxTotalConnections(config.getIntConfig(
                "http.ConnectionCount", 100));
        server.setMaxRetries(config.getIntConfig("http.retryTimes", 2));
        // 设置是否使用代理
        if (config.getBooleanConfig("http.useProxy", false))
        {
            HttpClient client = server.getHttpClient();
            HttpHost proxy = new HttpHost(config.getConfig("http.proxyIp"),
                    config.getIntConfig("http.proxyPort"));
            client.getParams().setParameter(ConnRoutePNames.DEFAULT_PROXY,
                    proxy);
            if (client instanceof DefaultHttpClient)
            {
                String user = config.getConfig("http.proxyUser");
                String pwd = config.getConfig("http.proxyPassword");
                if (StringUtils.isNotEmpty(user) && StringUtils.isNotEmpty(pwd))
                {
                    Credentials credentials = new NTCredentials(user, pwd, "",
                            "");
                    ((DefaultHttpClient) client).getCredentialsProvider()
                            .setCredentials(AuthScope.ANY, credentials);
                }
            }
        }
    }

    /**
     * 对响应返回的高亮结果和原始字段进行document合并
     * 
     * @param rsp
     * @param keyField
     *            主键字段的名称
     * @param resultMaps
     *            如果此参数为空，则直接将高亮直接合并至原始document,否则新生成一个Map
     */
    public static void mergeHighLight(QueryResponse rsp, String keyField,
            List<Map<String, Object>> resultMaps)
    {
        SolrDocumentList rspList = rsp.getResults();
        if (CollectionUtils.isEmpty(rspList)
                || MapUtils.isEmpty(rsp.getHighlighting()))
        {
            return;
        }
        Object key = null;
        Map<String, Object> result = null;
        boolean useSource = resultMaps == null;
        for (SolrDocument doc : rspList)
        {
            if (useSource)
            {
                result = doc;
            }
            else
            {
                result = new HashMap<String, Object>();
            }
            key = doc.get(keyField);
            Map<String, List<String>> map = rsp.getHighlighting().get(key);
            if (MapUtils.isNotEmpty(map))
            {
                if (!useSource)
                {
                    result.putAll(doc);
                }
                for (Entry<String, List<String>> en : map.entrySet())
                {
                    if (en.getValue().size() == 1)
                    {
                        result.put(en.getKey(), en.getValue().get(0));
                    }
                    else
                    {
                        result.put(en.getKey(), en.getValue());
                    }
                }
            }
            if (!useSource)
            {
                resultMaps.add(result);
            }
        }
    }

    /**
     * 关闭当前搜索器
     */
    void close()
    {
        server.shutdown();
    }
}
