package cn.pconline.search.common.util;

//${license}

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import org.apache.commons.io.IOUtils;

import cn.pconline.search.common.Configuration;

/**
 * 默认读取的是classpath目录下的 {@link Config#CONFIG_FILE}文件
 * 
 * @author zengjie
 * @since 2013-9-24
 * @see
 */
public class Config implements Configuration
{

    /**
     * 默认读取的配置文件
     */
    public static final String CONFIG_FILE = "search.properties";

    private Properties config = new Properties();

    private String configFile;

    /**
     * 实用工具类，不能建立实例
     */
    public Config(String configFile)
    {
        this.configFile = configFile;
        InputStream is = null;
        try
        {
            is = Config.class.getClassLoader().getResourceAsStream(configFile);
            if (is == null)
            {
                throw new RuntimeException("无法再当前classpath中找到配置文件："
                        + configFile + " !");
            }
            config.load(is);
        }
        catch (IOException e)
        {
            throw new RuntimeException("读取配置文件" + configFile + "失败!", e);
        }
        finally
        {
            IOUtils.closeQuietly(is);
        }
    }

    @Override
    public String getConfig(String key, String def)
    {
        String v = config.getProperty(key);
        if (v == null)
        {
            return def;
        }
        return v;
    }

    @Override
    public Integer getIntConfig(String key, Integer def)
    {
        String v = getConfig(key, null);
        if (v == null)
        {
            return def;
        }
        try
        {
            return Integer.parseInt(v);
        }
        catch (NumberFormatException e)
        {
            return def;
        }
    }

    @Override
    public Boolean getBooleanConfig(String key, Boolean def)
    {
        String v = getConfig(key, null);
        if (v == null)
        {
            return def;
        }
        return Boolean.parseBoolean(v);
    }

    private static Config defaultConf;

    /**
     * 获取唯一的默认配置对象
     * 
     * @see Config#CONFIG_FILE
     * @return
     */
    public static Config getDefaultConfig()
    {
        if (defaultConf == null)
        {
            synchronized (Config.class)
            {
                if (defaultConf != null)
                {
                    return defaultConf;
                }
                defaultConf = new Config(CONFIG_FILE);
            }
        }
        return defaultConf;
    }

    @Override
    public String getConfig(String key)
    {
        return config.getProperty(key);
    }

    @Override
    public Integer getIntConfig(String key)
    {
        try
        {
            return Integer.parseInt(getConfig(key));
        }
        catch (NumberFormatException e)
        {
            return null;
        }
    }

    @Override
    public Boolean getBooleanConfig(String key)
    {
        return Boolean.parseBoolean(getConfig(key));
    }

    @Override
    public String toString()
    {
        return "Config from file[" + configFile + "]";
    }
}
