package cn.pconline.search.common.util;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;

/**
 * url解析组合工具
 * 
 * @author zengjie
 * @since 2013-9-3
 * @see
 */
public class URLHolder implements Cloneable
{

    private Params params = new Params();

    private String host;

    private int port;

    private String requestUri;

    private String paramsEncoding = "gbk";

    public URLHolder()
    {
    }

    public URLHolder(String host, int port, String requestUri)
    {
        super();
        this.host = host;
        this.port = port;
        this.requestUri = requestUri;
    }

    public URLHolder(String host, int port, String requestUri,
            String paramsEncoding)
    {
        super();
        this.host = host;
        this.port = port;
        this.requestUri = requestUri;
        this.paramsEncoding = paramsEncoding;
    }

    /**
     * 查询参数url编码，默认为gbk
     * 
     * @return
     */
    public String getParamsEncoding()
    {
        return paramsEncoding;
    }

    /**
     * 查询参数url编码，默认为gbk
     * 
     * @param paramsEncoding
     */
    public void setParamsEncoding(String paramsEncoding)
    {
        this.paramsEncoding = paramsEncoding;
    }

    public Params getParams()
    {
        return params;
    }

    /**
     * 设置当前URL的查询字符串
     * 
     * @param paramString
     */
    public void setParamString(String paramString)
    {
        if (StringUtils.isBlank(paramString))
        {
            return;
        }
        String[] kvPair = paramString.split("\\s?\\&\\s?");
        if (kvPair == null || kvPair.length == 0)
        {
            return;
        }
        String value = null;
        for (String kv : kvPair)
        {
            String[] arr = kv.split("\\s?=\\s?", 2);
            if (arr != null && arr.length == 2
                    && StringUtils.isNotBlank(arr[0])
                    && StringUtils.isNotBlank(arr[1]))
            {
                try
                {
                    value = URLDecoder.decode(arr[1].trim(), paramsEncoding);
                }
                catch (UnsupportedEncodingException e)
                {
                    continue;
                }
                params.setParam(arr[0].trim(), value, false);
            }
        }
    }

    public String getHost()
    {
        return host;
    }

    public void setHost(String host)
    {
        this.host = host;
    }

    public int getPort()
    {
        return port;
    }

    public void setPort(int port)
    {
        this.port = port;
    }

    public String getRequestUri()
    {
        return requestUri;
    }

    public void setRequestUri(String requestUri)
    {
        this.requestUri = requestUri;
    }

    public URLHolder copy()
    {
        URLHolder holder = new URLHolder(host, port, requestUri, paramsEncoding);
        holder.params = (Params) params.clone();
        return holder;
    }

    @Override
    public String toString()
    {
        StringBuilder builder = new StringBuilder();
        builder.append("http://").append(host);
        if (port != 80)
        {
            builder.append(":").append(port);
        }
        if (!requestUri.startsWith("/"))
        {
            builder.append("/");
        }
        builder.append(requestUri);
        String paramStr = null;
        try
        {
            paramStr = params.toString(true);
        }
        catch (UnsupportedEncodingException e)
        {
        }
        if (StringUtils.isNotEmpty(paramStr))
        {
            builder.append("?").append(paramStr);
        }
        return builder.toString();
    }

    /**
     * 获取当前请求的URL路径对象
     * 
     * @param req
     * @return
     */
    public static URLHolder fromHttpRequest(HttpServletRequest req)
    {
        String pageUri = req.getRequestURI();
        if (pageUri == null)
        {
            pageUri = "/";
        }
        URLHolder holder = new URLHolder(req.getServerName(),
                req.getServerPort(), pageUri);
        @SuppressWarnings("unchecked")
        Map<String, String[]> params = req.getParameterMap();
        if (params == null)
        {
            return holder;
        }
        for (Entry<String, String[]> en : params.entrySet())
        {
            holder.getParams().put(en.getKey(), en.getValue());
        }
        return holder;
    }

    public class Params extends LinkedHashMap<String, String[]>
    {

        private static final long serialVersionUID = -890157917077460545L;

        Params()
        {
        }

        private Params(int capacity)
        {
            super(capacity);
        }

        public URLHolder setParam(String key, String value, boolean replace)
        {
            String[] values = this.get(key);
            if (values == null || replace)
            {
                values = new String[1];
            }
            else
            {
                String[] temp = new String[values.length + 1];
                System.arraycopy(values, 0, temp, 0, values.length);
                values = temp;
            }
            values[values.length - 1] = value;
            this.put(key, values);
            return URLHolder.this;
        }

        public URLHolder clearAllParams()
        {
            super.clear();
            return URLHolder.this;
        }

        public URLHolder removeParam(String key)
        {
            super.remove(key);
            return URLHolder.this;
        }

        public URLHolder addOrReplace(String key, Object... values)
        {
            if (ArrayUtils.isNotEmpty(values))
            {
                String[] vs = new String[values.length];
                for (int i = 0; i < vs.length; i++)
                {
                    vs[i] = values[i].toString();
                }
                super.put(key, vs);
            }
            return URLHolder.this;
        }

        public String toString(boolean encode)
                throws UnsupportedEncodingException
        {
            StringBuilder builder = new StringBuilder();
            for (java.util.Map.Entry<String, String[]> en : entrySet())
            {
                for (String value : en.getValue())
                {
                    builder.append(en.getKey());
                    builder.append("=");
                    if (encode)
                    {
                        builder.append(URLEncoder.encode(value, paramsEncoding));
                    }
                    else
                    {
                        builder.append(value);
                    }
                    builder.append("&");
                }
            }
            if (builder.length() > 0)
            {
                builder.delete(builder.length() - 1, builder.length());
            }
            return builder.toString();
        }

        @Override
        public Object clone()
        {
            Params params = new Params(this.size());
            String[] arr;
            for (java.util.Map.Entry<String, String[]> en : entrySet())
            {
                arr = new String[en.getValue().length];
                System.arraycopy(en.getValue(), 0, arr, 0, en.getValue().length);
                params.put(en.getKey(), arr);
            }
            return params;
        }

        @Override
        public String toString()
        {
            try
            {
                return toString(false);
            }
            catch (UnsupportedEncodingException e)
            {
                return null;
            }
        }
    }
}
