package cn.pconline.search.common.query;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.collections.CollectionUtils;

import cn.pconline.search.common.util.QueryUtil;

/**
 * 表示一个查询条件
 * 
 * @author zengjie
 * @since 2013-9-24
 * @see
 */
public class Condition implements Iterable<Condition>
{

    private String field;

    private Object[] value;

    private Operator operator = Operator.SHOULD;

    private List<Condition> childs;

    private Float boost;

    public Condition()
    {
    }

    public Condition(Operator operator)
    {
        this.operator = operator;
    }

    /**
     * 构造函数
     * 
     * @param field
     *            匹配的字段
     * @param operator
     *            匹配的操作符
     * @param value
     *            匹配的值，可以是多个，代表匹配其中的某一个值即可
     */
    public Condition(String field, Operator operator, Object... value)
    {
        super();
        this.field = field;
        this.value = value;
        this.operator = operator;
    }

    public Condition(String field, Operator operator, Float boost,
            Object... value)
    {
        super();
        this.field = field;
        this.value = value;
        this.operator = operator;
        this.boost = boost;
    }

    public Condition(String field, Object... value)
    {
        super();
        this.field = field;
        this.value = value;
    }

    public Condition(String field, Float boost, Object... value)
    {
        super();
        this.field = field;
        this.value = value;
        this.boost = boost;
    }

    public void addChild(Condition c)
    {
        if (childs == null)
        {
            childs = new ArrayList<Condition>();
        }
        childs.add(c);
    }

    @SuppressWarnings("unchecked")
    @Override
    public Iterator<Condition> iterator()
    {
        if (childs == null)
        {
            return Collections.EMPTY_LIST.iterator();
        }
        return childs.iterator();
    }

    public boolean hasChild()
    {
        return CollectionUtils.isNotEmpty(childs);
    }

    public String getField()
    {
        return field;
    }

    public void setField(String field)
    {
        this.field = field;
    }

    public void setValue(String... value)
    {
        this.value = value;
    }

    /**
     * 默认为 {@link Operator#SHOULD}
     * 
     * @return
     */
    public Operator getOperator()
    {
        return operator;
    }

    public void setOperator(Operator operator)
    {
        if (operator == null)
        {
            return;
        }
        this.operator = operator;
    }

    public Float getBoost()
    {
        return boost;
    }

    public void setBoost(Float boost)
    {
        this.boost = boost;
    }

    @Override
    public String toString()
    {
        StringBuilder sb = new StringBuilder();
        sb.append(operator);
        if (hasChild())
        {

            sb.append("(");
            for (Condition c : this)
            {
                sb.append(c).append(" ");
            }
            sb.append(")");
        }
        else if (value != null && value.length > 0)
        {
            sb.append(field);
            sb.append(":");
            sb.append("(");
            for (Object obj : value)
            {
                sb.append(QueryUtil.escapeQStr(obj.toString()));
                sb.append(" ");
            }
            sb.append(")");
        }
        if (boost != null && boost >= 0 && boost.floatValue() != 1F)
        {
            sb.append("^");
            sb.append(formatFloat(boost));
        }
        return sb.toString();
    }

    private static String formatFloat(float f)
    {
        String s = String.valueOf(f);
        int i = 0;
        if ((i = s.indexOf('E')) > 0)
        {
            StringBuilder sb = new StringBuilder();
            boolean negative = s.charAt(i + 1) == '-';
            int eCount = Integer
                    .parseInt(s.substring(negative ? i + 2 : i + 1));
            String base = s.substring(0, i);
            if (negative)
            {
                sb.append("0.");
                int count = 0;
                int last = eCount;
                if (base.indexOf('.') >= 0)
                {
                    count = eCount - base.indexOf('.');
                    last = Integer
                            .parseInt(base.substring(base.indexOf('.') + 1));
                    base = base.substring(0, base.indexOf('.'));
                }
                for (int j = 0; j < count; j++)
                {
                    sb.append("0");
                }
                sb.append(base);
                if (last > 0)
                {
                    sb.append(last);
                }
            }
            else
            {
                int count = eCount;
                if (base.indexOf('.') >= 0)
                {
                    count = eCount - (base.length() - base.indexOf('.') - 1);
                    sb.append(base.substring(0, base.indexOf('.')));
                    sb.append(base.substring(base.indexOf('.') + 1));
                }
                for (int j = 0; j < count; j++)
                {
                    sb.append("0");
                }
                sb.append(".0");
            }
            return sb.toString();
        }
        else
        {
            return s;
        }
    }

    public static void main(String[] args)
    {
        Q q = new Q();
        Condition c = new Condition("status", Operator.MUST, "AND", "AND",
                "AND", "AND");
        c.setBoost(200000000F);
        q.addCondition(c);
        System.out.println(q);
    }
}
