package cn.pconline.search.common;

import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.UnknownHostException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Enumeration;

import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import cn.pconline.search.common.log.SearchLogger;
import cn.pconline.search.common.tools.ad.AdManager;
import cn.pconline.search.common.tools.homonym.Homonyms;
import cn.pconline.search.common.tools.sensitive.AnalyzeSensitiveFilter;
import cn.pconline.search.common.tools.sensitive.SensitiveFilter;
import cn.pconline.search.common.util.Mailer;

/**
 * 搜索框架
 * 
 * @author zengjie
 * @since 2013-9-22
 * @see
 */
public class SearchFrame
{

    private volatile boolean inited = false;

    private static Logger logger = Logger.getLogger(SearchFrame.class);

    protected Configuration configuration;

    private SearchLogger searchLogger;

    private SolrSearcher searcher;

    private Homonyms homonyms;

    private SensitiveFilter sensitiveFilter;

    private AdManager adManager;

    /**
     * 对框架通过指定的配置进行初始化
     * 
     * @param config
     */
    public void init(Configuration config)
    {

        if (!inited)
        {
            synchronized (this)
            {
                if (inited)
                {
                    return;
                }
                long start = System.currentTimeMillis();
                logger.debug("Init SearchFrame with config{" + config + "}");
                this.configuration = config;
                // 是否开启搜索日志记录,默认关闭
                if (config.getBooleanConfig("enableSearchLog", false))
                {
                    searchLogger = new SearchLogger(config);
                }
                searcher = new SolrSearcher(config);
                // 初始化同音词(如果配置中存在同音词典文件)
                String homonymsFiles = config.getConfig("homonymsFiles", null);
                if (StringUtils.isNotBlank(homonymsFiles))
                {
                    homonyms = new Homonyms(homonymsFiles);
                }
                // 初始化敏感词，默认不加载
                if (config.getBooleanConfig("enbaleSensitiveFilter", false))
                {
                    sensitiveFilter = new AnalyzeSensitiveFilter(config);
                    logger.debug("finish load SensitiveFilter");
                }
                // 默认不启用广告加载器
                if (config.getBooleanConfig("enableAdManager", false))
                {
                    adManager = new AdManager(config);
                    logger.debug("finish load AdManager");
                }

                logger.debug("SearchFrame init complete in "
                        + (System.currentTimeMillis() - start) + " ms");
                inited = true;
            }
        }
    }

    private SearchFrame()
    {
    }

    private void checkInit()
    {
        if (!inited)
        {
            throw new IllegalStateException(
                    "Frame has not been initialize or has been closed,pelease make sure has invoked #init# method");
        }
    }

    private static SearchFrame instance = new SearchFrame();

    /**
     * 获取单例，使用前确保调用了init方法
     * 
     * @return
     */
    public static SearchFrame get()
    {
        return instance;
    }

    /**
     * 获取搜索日志记录器
     * <p>
     * <b>notic</b>:加载配置中enableSearchLog参数可以控制是否关闭搜索日志记录
     * </p>
     * 
     * @return
     */
    public SearchLogger getLogger()
    {
        checkInit();
        if (searchLogger == null)
        {
            throw new IllegalStateException("SearchLogger is not enable");
        }
        return searchLogger;
    }

    /**
     * 获取对solr进行访问的Searcher
     * 
     * @return
     */
    public SolrSearcher getSearcher()
    {
        checkInit();
        return searcher;
    }

    /**
     * 获取当前框架上的同音词典
     * 
     * @return
     */
    public Homonyms getHomonyms()
    {
        checkInit();
        return homonyms;
    }

    /**
     * 获取同音词
     * 
     * @see SearchFrame#getHomonyms()
     * @param key
     * @return
     */
    public String getHomonynsWords(String key)
    {
        checkInit();
        if (homonyms == null)
        {
            return "";
        }
        return homonyms.getHomonyms(key);
    }

    /**
     * 获取当前框架敏感词过滤器
     * <p>
     * <p>
     * 在 {@link SearchFrame#init(Configuration)}方法中需要的Configuration配置项为
     * </p>
     * <table border="1">
     * <tr>
     * <td align="center"><b>配置key</b></td>
     * <td align="center"><b>说明</b></td>
     * <td align="center"><b>可选值</b></td>
     * <td align="center"><b>必要</b></td>
     * </tr>
     * <tr>
     * <td>enbaleSensitiveFilter</td>
     * <td>是否加载敏感词过滤器</td>
     * <td>true|false(默认为true)</td>
     * <td>false</td>
     * </tr>
     * <tr>
     * <td>sensitiveWordDb</td>
     * <td>敏感词加载数据库jdbc连接字符串</td>
     * <td>oracle数据库</td>
     * <td>true</td>
     * </tr>
     * <tr>
     * <td>sensitiveFilterDomain</td>
     * <td>指定是要加载哪个网的敏感词</td>
     * <td>六网域名 pconline、pcauto...</td>
     * <td>true</td>
     * </tr>
     * <tr>
     * <td>sensitiveWordReloadTime</td>
     * <td>敏感词每日重新加载时间</td>
     * <td>格式为00:00:00(默认为当前时间)</td>
     * <td>false</td>
     * </tr>
     * </table>
     * </p>
     * 
     * @return
     */
    public SensitiveFilter getSensitiveFilter()
    {
        checkInit();
        if (sensitiveFilter == null)
        {
            throw new IllegalStateException(
                    "sensitive filter is not enable or load fail");
        }
        return sensitiveFilter;
    }

    /**
     * 关闭当前搜索框架对象,所有获取操作立即不可用
     */
    public void shutdown()
    {
        if (!inited)
        {
            return;
        }
        try
        {
            searcher.close();
            searcher = null;
            if (searchLogger != null)
            {
                searchLogger.close();
            }
            searchLogger = null;
            homonyms = null;
        }
        finally
        {
            inited = false;
        }
    }

    /**
     * 获取广告管理器
     * 
     * @return
     */
    public AdManager getAdManager()
    {
        checkInit();
        if (adManager == null)
        {
            throw new IllegalStateException("AdManager isn't enable!");
        }
        return adManager;
    }

    /**
     * 发送错误邮件
     * 
     * @param title
     * @param errInfo
     */
    public void sendErrMail(String title, String errInfo)
    {
        checkInit();
        StringBuilder sb = new StringBuilder();
        sb.append("<html><head></head><body><div><h1>环境信息:</h1><ul>");
        sb.append("<li>服务器:" + getLocalAddress() + "</li>");
        sb.append("<li>时间:"
                + new SimpleDateFormat("yyyy-MM-dd HH:mm:ss")
                        .format(new Date()) + "</li>");
        sb.append("<li></li>");
        sb.append("</ul></div>");
        sb.append("<div><h1>错误信息</h1><div>");
        sb.append(errInfo);
        sb.append("</div></div></body></html>");
        Mailer.sendMail(title, sb.toString(),
                configuration.getConfig("mailServer"),
                configuration.getConfig("mailFrom"),
                configuration.getConfig("mailTo"),
                configuration.getConfig("mailUser"),
                configuration.getConfig("mailPassword"));
    }

    /**
     * 获取当前搜索框架是否可用
     * 
     * @return
     */
    public boolean isAvaliable()
    {
        return inited;
    }

    /**
     * 获取本机在当前局域网中的可用IP
     * 
     * @return
     */
    public static String getLocalAddress()
    {
        Enumeration<NetworkInterface> netInterfaces = null;
        try
        {
            netInterfaces = NetworkInterface.getNetworkInterfaces();
            while (netInterfaces.hasMoreElements())
            {
                NetworkInterface ni = netInterfaces.nextElement();
                Enumeration<InetAddress> ips = ni.getInetAddresses();
                while (ips.hasMoreElements())
                {
                    String ip = ips.nextElement().getHostAddress();
                    if (ip.startsWith("192.") || ip.startsWith("10.")
                            || ip.startsWith("172."))
                    {
                        return ip;
                    }
                }
            }
        }
        catch (Exception e)
        {
            logger.error("Try get local LAN IP error!", e);
        }
        try
        {
            return InetAddress.getLocalHost().getHostAddress();
        }
        catch (UnknownHostException e)
        {
            logger.error("Try get local LAN IP error!", e);
            return null;
        }
    }
}
