package cn.pconline.search.common.util;

import java.math.BigDecimal;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang.StringUtils;

public class QueryUtil
{

    /**
     * 根据指定关键字和权值Map转换成solr内嵌edismax查询
     * 
     * @param keyword
     * @param boostMap
     * @return
     */
    public static String toDismaxQuery(String keyword,
            Map<String, Float> boostMap)
    {
        StringBuilder builder = new StringBuilder();
        builder.append("_query_:\"{!edismax qf='");
        for (Entry<String, Float> en : boostMap.entrySet())
        {
            builder.append(en.getKey()).append("^")
                    .append(new BigDecimal(en.getValue()).toPlainString())
                    .append(" ");
        }
        builder.append("'}").append(escapeQStr(keyword));
        return builder.toString();
    }

    private static Map<Pattern, String> escapeMap = new LinkedHashMap<Pattern, String>();

    private static void addToEscapeMap(String key, String value)
    {
        escapeMap.put(Pattern.compile(key, Pattern.LITERAL),
                Matcher.quoteReplacement(value));
    }

    static
    {
        // 一般转义
        addToEscapeMap("\\", "\\\\");
        addToEscapeMap("\"", "\\\"");
        addToEscapeMap("/", "\\/");
        // LUCENE特殊字符
        addToEscapeMap("?", "\\?");
        addToEscapeMap("*", "\\*");
        addToEscapeMap(":", "\\:");
        addToEscapeMap("~", "\\~");
        addToEscapeMap("+", "\\+");
        addToEscapeMap("-", "\\-");
        addToEscapeMap("^", "\\^");
        // 各种括号
        addToEscapeMap("(", "\\(");
        addToEscapeMap(")", "\\)");
        addToEscapeMap("[", "\\[");
        addToEscapeMap("]", "\\]");
        addToEscapeMap("{", "\\{");
        addToEscapeMap("}", "\\}");
        // LUCENE查询连接符
        addToEscapeMap("OR", "\\OR");
        addToEscapeMap("AND", "\\AND");
        addToEscapeMap("NOT", "\\NOT");
    }

    /**
     * 转义查询中的特殊字符串
     * 
     * @param query
     * @return
     */
    public static String escapeQStr(String query)
    {
        if (StringUtils.isEmpty(query))
        {
            return query;
        }
        query = query.trim();
        for (Entry<Pattern, String> e : escapeMap.entrySet())
        {
            query = e.getKey().matcher(query).replaceAll(e.getValue());
        }
        return query;
    }

    public static void main(String[] args)
    {
        System.out.println(escapeQStr("\"-+=\\/?<>*ANDORANDOR#$%^@#"));
    }

    public static boolean shouldUsePharse(String input)
    {
        if (StringUtils.isBlank(input))
        {
            return false;
        }
        input = input.trim();
        for (int i = 0; i < input.length(); i++)
        {
            char c = input.charAt(i);
            if (Character.isSpaceChar(input.charAt(i))
                    || Character.isWhitespace(input.charAt(i)))
            {
                return false;
            }
            if (!Character.isDigit(input.charAt(i))
                    && !Character.isUpperCase(c) && !Character.isLowerCase(c)
                    && input.charAt(i) != '_' && input.charAt(i) != '-')
            {
                return false;
            }
        }
        return true;
    }

}
