package cn.pconline.search.common.query;

import java.io.IOException;
import java.io.InputStream;
import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.NameValuePair;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.entity.mime.FormBodyPart;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntity;
import org.apache.http.entity.mime.content.InputStreamBody;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.message.BasicHeader;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.apache.solr.client.solrj.ResponseParser;
import org.apache.solr.client.solrj.SolrRequest;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.impl.HttpSolrServer;
import org.apache.solr.client.solrj.request.RequestWriter;
import org.apache.solr.client.solrj.util.ClientUtils;
import org.apache.solr.common.SolrException;
import org.apache.solr.common.params.CommonParams;
import org.apache.solr.common.params.ModifiableSolrParams;
import org.apache.solr.common.params.SolrParams;
import org.apache.solr.common.util.ContentStream;
import org.apache.solr.common.util.NamedList;

import cn.pconline.r.client.RequestCallback;
import cn.pconline.r.client.ResponseExtractor;
import cn.pconline.r.client.SimpleHttpTemplate;

/**
 * 基于R系统的http请求
 * 
 * @author zengjie
 * @since 2013-10-17
 * @see
 */
public class RHttpSolrServer extends HttpSolrServer
{

    /**
     * 
     */
    private static final long serialVersionUID = -1101254915951168009L;

    private SimpleHttpTemplate httpTemplate;

    public RHttpSolrServer(String baseURL, SimpleHttpTemplate httpTemplate)
    {
        super(baseURL);
        this.httpTemplate = httpTemplate;
    }

    @Override
    public NamedList<Object> request(SolrRequest request,
            final ResponseParser processor) throws SolrServerException,
            IOException
    {
        if (httpTemplate == null)
        {
            return super.request(request, processor);
        }
        try
        {
            SolrParams sourceParams = request.getParams();
            final Collection<ContentStream> streams = requestWriter
                    .getContentStreams(request);
            String path = requestWriter.getPath(request);
            if (path == null || !path.startsWith("/"))
            {
                path = "/select";
            }

            ResponseParser parser = request.getResponseParser();
            if (parser == null)
            {
                parser = this.parser;
            }
            ModifiableSolrParams wparams = new ModifiableSolrParams(
                    sourceParams);
            if (parser != null)
            {
                wparams.set(CommonParams.WT, parser.getWriterType());
                wparams.set(CommonParams.VERSION, parser.getVersion());
            }
            if (invariantParams != null)
            {
                wparams.add(invariantParams);
            }
            final SolrParams params = wparams;
            String requestUrl = baseUrl + path;
            if (SolrRequest.METHOD.GET == request.getMethod())
            {
                if (streams != null)
                {
                    throw new SolrException(
                            SolrException.ErrorCode.BAD_REQUEST,
                            "GET can't send streams!");
                }
                requestUrl += ClientUtils.toQueryString(params, false);
            }
            final String url = requestUrl;

            ResponseExtractor<NamedList<Object>> extractor = new ResponseExtractor<NamedList<Object>>()
            {

                @SuppressWarnings("deprecation")
                public NamedList<Object> extractData(HttpResponse response)
                        throws IOException
                {
                    boolean shouldClose = true;
                    InputStream respBody = null;
                    try
                    {
                        int httpStatus = response.getStatusLine()
                                .getStatusCode();
                        switch (httpStatus)
                        {
                        case HttpStatus.SC_OK:
                        case HttpStatus.SC_BAD_REQUEST:
                        case HttpStatus.SC_CONFLICT: // 409
                            break;
                        case HttpStatus.SC_MOVED_PERMANENTLY:
                        case HttpStatus.SC_MOVED_TEMPORARILY:
                            break;
                        default:
                            throw new SolrException(
                                    SolrException.ErrorCode
                                            .getErrorCode(httpStatus),
                                    "Server at "
                                            + getBaseURL()
                                            + " returned non ok status:"
                                            + httpStatus
                                            + ", message:"
                                            + response.getStatusLine()
                                                    .getReasonPhrase());

                        }
                        respBody = response.getEntity().getContent();
                        if (processor == null)
                        {
                            // no processor specified, return raw stream
                            NamedList<Object> rsp = new NamedList<Object>();
                            rsp.add("stream", respBody);
                            // Only case where stream should not be closed
                            shouldClose = false;
                            return rsp;
                        }
                        String charset = EntityUtils.getContentCharSet(response
                                .getEntity());
                        NamedList<Object> rsp = processor.processResponse(
                                respBody, charset);
                        if (httpStatus != HttpStatus.SC_OK)
                        {
                            String reason = null;
                            try
                            {
                                NamedList<?> err = (NamedList<?>) rsp
                                        .get("error");
                                if (err != null)
                                {
                                    reason = (String) err.get("msg");
                                    // TODO? get the trace?
                                }
                            }
                            catch (Exception ex)
                            {
                            }
                            if (reason == null)
                            {
                                StringBuilder msg = new StringBuilder();
                                msg.append(response.getStatusLine()
                                        .getReasonPhrase());
                                msg.append("\n\n");
                                msg.append("request: " + url);
                                reason = java.net.URLDecoder.decode(
                                        msg.toString(), "UTF-8");
                            }
                            throw new SolrException(
                                    SolrException.ErrorCode
                                            .getErrorCode(httpStatus),
                                    reason);
                        }
                        return rsp;
                    }
                    finally
                    {
                        if (respBody != null && shouldClose)
                        {
                            try
                            {
                                respBody.close();
                            }
                            catch (Throwable t)
                            {
                            } // ignore
                        }
                    }
                }
            };

            if (SolrRequest.METHOD.GET == request.getMethod())
            {
                return httpTemplate.get(requestUrl, null, extractor);
            }
            else
            {
                final HttpEntity[] entityHolder = new HttpEntity[1];
                RequestCallback callback = new RequestCallback()
                {

                    public void doWithRequest(HttpUriRequest request)
                            throws IOException
                    {

                        boolean isMultipart = (streams != null && streams
                                .size() > 1);

                        LinkedList<NameValuePair> postParams = new LinkedList<NameValuePair>();
                        HttpPost post = (HttpPost) request;
                        if (streams == null || isMultipart)
                        {
                            post.setHeader("Content-Charset", "UTF-8");
                            if (!isMultipart)
                            {
                                post.addHeader("Content-Type",
                                        "application/x-www-form-urlencoded; charset=UTF-8");
                            }

                            List<FormBodyPart> parts = new LinkedList<FormBodyPart>();
                            Iterator<String> iter = params
                                    .getParameterNamesIterator();
                            while (iter.hasNext())
                            {
                                String p = iter.next();
                                String[] vals = params.getParams(p);
                                if (vals != null)
                                {
                                    for (String v : vals)
                                    {
                                        if (isMultipart)
                                        {
                                            parts.add(new FormBodyPart(p,
                                                    new StringBody(v, Charset
                                                            .forName("UTF-8"))));
                                        }
                                        else
                                        {
                                            postParams
                                                    .add(new BasicNameValuePair(
                                                            p, v));
                                        }
                                    }
                                }
                            }

                            if (isMultipart)
                            {
                                for (ContentStream content : streams)
                                {
                                    String contentType = content
                                            .getContentType();
                                    if (contentType == null)
                                    {
                                        contentType = "application/octet-stream"; // default
                                    }
                                    parts.add(new FormBodyPart(content
                                            .getName(), new InputStreamBody(
                                            content.getStream(), contentType,
                                            content.getName())));
                                }
                            }

                            if (parts.size() > 0)
                            {
                                MultipartEntity entity = new MultipartEntity(
                                        HttpMultipartMode.STRICT);
                                for (FormBodyPart p : parts)
                                {
                                    entity.addPart(p);
                                }
                                entityHolder[0] = entity;
                            }
                            else
                            {
                                entityHolder[0] = new UrlEncodedFormEntity(
                                        postParams, "UTF-8");
                            }
                        }
                        // It is has one stream, it is the post body, put the
                        // params
                        // in the URL
                        else
                        {
                            String pstr = ClientUtils.toQueryString(params,
                                    false);
                            try
                            {
                                post.setURI(new URI(url + pstr));
                            }
                            catch (URISyntaxException e)
                            {
                            }

                            // Single stream as body
                            // Using a loop just to get the first one
                            final ContentStream[] contentStream = new ContentStream[1];
                            for (ContentStream content : streams)
                            {
                                contentStream[0] = content;
                                break;
                            }
                            if (contentStream[0] instanceof RequestWriter.LazyContentStream)
                            {
                                entityHolder[0] = new InputStreamEntity(
                                        contentStream[0].getStream(), -1)
                                {

                                    public Header getContentType()
                                    {
                                        return new BasicHeader("Content-Type",
                                                contentStream[0]
                                                        .getContentType());
                                    }

                                    public boolean isRepeatable()
                                    {
                                        return false;
                                    }

                                };
                            }
                            else
                            {
                                entityHolder[0] = new InputStreamEntity(
                                        contentStream[0].getStream(), -1)
                                {

                                    public Header getContentType()
                                    {
                                        return new BasicHeader("Content-Type",
                                                contentStream[0]
                                                        .getContentType());
                                    }

                                    public boolean isRepeatable()
                                    {
                                        return false;
                                    }
                                };
                            }
                        }
                    }
                };
                return httpTemplate.post(url, null, callback, extractor,
                        entityHolder[0]);
            }
        }
        catch (ConnectException e)
        {
            throw new SolrServerException("Server refused connection at: "
                    + getBaseURL(), e);
        }
        catch (SocketTimeoutException e)
        {
            throw new SolrServerException(
                    "Timeout occured while waiting response from server at: "
                            + getBaseURL(), e);
        }
        catch (IOException e)
        {
            throw new SolrServerException(
                    "IOException occured when talking to server at: "
                            + getBaseURL(), e);
        }
    }

    @Override
    public void shutdown()
    {
        if (httpTemplate != null)
        {
            httpTemplate.shutdown();
        }
        super.shutdown();
    }

    @Override
    public void setSoTimeout(int timeout)
    {
        if (httpTemplate != null)
        {
            httpTemplate.setReadTimeout(timeout);
        }
        super.setSoTimeout(timeout);
    }

}
