package cn.pconline.search.common.tools.ad;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.locks.ReentrantLock;

import oracle.jdbc.OracleDriver;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import cn.pconline.search.common.Configuration;
import cn.pconline.search.common.query.SolrQueryExt;
import cn.pconline.search.common.util.DailyFixTimeRunThread;

/**
 * 重写的广告管理器
 * 
 * @author zengjie
 * @since 2013-10-25
 * @see
 */
public class AdManager extends DailyFixTimeRunThread
{

    private static Logger logger = Logger.getLogger(AdManager.class);

    private ReentrantLock loadLock = new ReentrantLock();

    private String adDBUrl;

    private String dbUser;

    private String dbPwd;

    private volatile Map<String, Map<String, List<AdRecord>>> adMap = null;

    public AdManager(Configuration config)
    {
        super(config.getConfig("adReloadTime"), "Ad-Reload-Thread");
        this.adDBUrl = config.getConfigNotEmpty("ad.dburl");
        this.dbUser = config.getConfigNotEmpty("ad.user");
        this.dbPwd = config.getConfigNotEmpty("ad.passwd");
        loadAd();
    }

    @Override
    protected void doRun()
    {
        loadAd();
    }

    public void loadAd()
    {
        loadLock.lock();
        Connection conn = null;
        Statement sta = null;
        ResultSet res = null;
        try
        {
            DriverManager.registerDriver(new OracleDriver());
            conn = DriverManager.getConnection(adDBUrl, dbUser, dbPwd);
            String sql = "SELECT SERVICE_ID, KEY_WORD, AD_CONTENT_ID, AD_DESC, AD_URL, weight "
                    + "FROM YISOU_AD "
                    + "WHERE START_DATE<=SYSDATE AND SYSDATE<=(END_DATE+1)";
            AdRecord record = null;
            sta = conn.createStatement();
            res = sta.executeQuery(sql);
            Map<String, Map<String, List<AdRecord>>> appAdMap = new HashMap<String, Map<String, List<AdRecord>>>();

            while (res.next())
            {
                record = new AdRecord();
                record.setApp(res.getString("SERVICE_ID"));
                record.setKey(res.getString("KEY_WORD").toUpperCase());
                record.setMappedDocId(res.getString("AD_CONTENT_ID"));
                record.setAdDesc(res.getString("AD_DESC"));
                record.setTargetUrl(res.getString("AD_URL"));
                String weight = res.getString("weight");
                record.setWeight(weight == null ? 0 : Float.parseFloat(weight));
                int idx = record.getMappedDocId() == null ? -1 : record
                        .getMappedDocId().indexOf(':');
                if (idx >= 0)
                {
                    record.setMappedDocId(record.getMappedDocId().substring(
                            idx + 1));
                }
                Map<String, List<AdRecord>> keyRecordMap = appAdMap.get(record
                        .getApp());
                if (keyRecordMap == null)
                {
                    keyRecordMap = new HashMap<String, List<AdRecord>>();
                    appAdMap.put(record.getApp(), keyRecordMap);
                }
                // 获取指定关键字的记录列表
                List<AdRecord> records = keyRecordMap.get(record.getKey());
                if (records == null)
                {
                    records = new ArrayList<AdRecord>();
                    keyRecordMap.put(record.getKey(), records);
                }
                records.add(record);
            }
            normalizeAdMap(appAdMap);
            adMap = appAdMap;
        }
        catch (Exception e)
        {
            logger.error("load ad info error", e);
        }
        finally
        {
            loadLock.unlock();
            if (res != null)
            {
                try
                {
                    res.close();
                }
                catch (SQLException e)
                {
                }
            }
            if (sta != null)
            {
                try
                {
                    sta.close();
                }
                catch (SQLException e)
                {
                }
            }
            if (conn != null)
            {
                try
                {
                    conn.close();
                }
                catch (SQLException e)
                {
                }
            }
        }
    }

    /**
     * 获取指定app和指定关键字的广告记录(已经排序ok)
     * 
     * @param app
     * @param keyword
     * @return
     */
    public List<AdRecord> getAdRecords(String app, String keyword)
    {
        if (StringUtils.isEmpty(app) || StringUtils.isBlank(keyword))
        {
            return null;
        }
        Map<String, List<AdRecord>> map = adMap.get(app);
        if (map == null)
        {
            return null;
        }
        List<AdRecord> records = map.get(keyword);
        if (records == null)
        {
            return null;
        }
        return Collections.unmodifiableList(records);
    }

    private static void normalizeAdMap(
            Map<String, Map<String, List<AdRecord>>> adMap)
    {
        for (Entry<String, Map<String, List<AdRecord>>> en : adMap.entrySet())
        {
            Map<String, List<AdRecord>> map = en.getValue();
            for (List<AdRecord> records : map.values())
            {
                Collections.sort(records, new Comparator<AdRecord>()
                {

                    public int compare(AdRecord o1, AdRecord o2)
                    {
                        if (o1.getWeight() > o2.getWeight())
                        {
                            return 1;
                        }
                        if (o1.getWeight() < o2.getWeight())
                        {
                            return -1;
                        }
                        return 0;
                    }
                });
            }
        }
    }

    /**
     * 将广告ID插入到指定的solr查询
     * 
     * @param query
     * @param records
     */
    public static void mergeAdIntoQuery(SolrQueryExt query,
            List<AdRecord> records)
    {
        if (CollectionUtils.isEmpty(records))
        {
            return;
        }
        String[] adKeys = new String[records.size()];
        for (int i = 0; i < adKeys.length; i++)
        {
            adKeys[i] = records.get(i).getMappedDocId();
        }
        query.addIncludeAdIds(adKeys);
    }

}
