package cn.pconline.search.common.util;

import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;

/**
 * 表示一个http地址对象，可以通过方法对地址进行修改
 * 
 * @author zengjie
 * @since 2013-9-3
 * @see
 */
public final class HttpUrl implements Cloneable, Serializable
{

    private static final long serialVersionUID = -6206126100089936640L;

    private Params params = new Params();

    private String host;

    private int port;

    private boolean isHttps = false;

    private String requestUri;

    private String paramsEncoding = "gbk";

    public HttpUrl()
    {
    }

    public HttpUrl(String host, int port, String requestUri)
    {
        super();
        this.host = host;
        this.port = port;
        this.requestUri = requestUri;
    }

    public HttpUrl(String host, int port, String requestUri,
            String paramsEncoding)
    {
        super();
        this.host = host;
        this.port = port;
        this.requestUri = requestUri;
        this.paramsEncoding = paramsEncoding;
    }

    /**
     * 查询参数url编码，默认为gbk
     * 
     * @return
     */
    public String getParamsEncoding()
    {
        return paramsEncoding;
    }

    /**
     * 查询参数url编码，默认为gbk
     * 
     * @param paramsEncoding
     */
    public void setParamsEncoding(String paramsEncoding)
    {
        this.paramsEncoding = paramsEncoding;
    }

    public Params getParams()
    {
        return params;
    }

    /**
     * 设置当前URL的查询字符串
     * 
     * @param paramString
     */
    public void setParamString(String paramString)
    {
        if (StringUtils.isBlank(paramString))
        {
            return;
        }
        String[] kvPair = paramString.split("\\s?\\&\\s?");
        if (kvPair == null || kvPair.length == 0)
        {
            return;
        }
        String value = null;
        for (String kv : kvPair)
        {
            String[] arr = kv.split("\\s?=\\s?", 2);
            if (arr != null && arr.length == 2
                    && StringUtils.isNotBlank(arr[0])
                    && StringUtils.isNotBlank(arr[1]))
            {
                try
                {
                    value = URLDecoder.decode(arr[1].trim(), paramsEncoding);
                }
                catch (UnsupportedEncodingException e)
                {
                    continue;
                }
                setParam(arr[0].trim(), value, false);
            }
        }
    }

    public String getHost()
    {
        return host;
    }

    public void setHost(String host)
    {
        this.host = host;
    }

    public int getPort()
    {
        return port;
    }

    public void setPort(int port)
    {
        this.port = port;
    }

    public String getRequestUri()
    {
        return requestUri;
    }

    public void setRequestUri(String requestUri)
    {
        this.requestUri = requestUri;
    }

    public boolean isHttps()
    {
        return isHttps;
    }

    public void setHttps(boolean isHttps)
    {
        this.isHttps = isHttps;
    }

    /**
     * 设置当前Url的参数
     * 
     * @param key
     * @param value
     * @param replace
     * @return
     */
    public HttpUrl setParam(String key, Object value, boolean replace)
    {
        String[] values = this.params.get(key);
        if (values == null || replace)
        {
            values = new String[1];
        }
        else
        {
            String[] temp = new String[values.length + 1];
            System.arraycopy(values, 0, temp, 0, values.length);
            values = temp;
        }
        values[values.length - 1] = value.toString();
        this.params.put(key, values);
        return HttpUrl.this;
    }

    /**
     * 获取参数
     * 
     * @param key
     * @return
     */
    public String getParam(String key)
    {
        String[] arr = this.params.get(key);
        if (arr != null && arr.length > 0)
        {
            return arr[0];
        }
        return null;
    }

    /**
     * 清除当前Url的参数
     * 
     * @return
     */
    public HttpUrl clearAllParams()
    {
        params.clear();
        return this;
    }

    /**
     * 清除指定参数
     * 
     * @param key
     * @return
     */
    public HttpUrl removeParam(String key)
    {
        params.remove(key);
        return this;
    }

    /**
     * 添加或者替换(如果当前存在指定key参数则进行替换)参数
     * 
     * @param key
     * @param values
     * @return
     */
    public HttpUrl replaceParam(String key, Object... values)
    {
        if (ArrayUtils.isNotEmpty(values))
        {
            if (values.length == 1 && values[0] == null)
            {
                return HttpUrl.this;
            }
            String[] vs = new String[values.length];
            for (int i = 0; i < vs.length; i++)
            {
                vs[i] = values[i].toString();
            }
            params.put(key, vs);
        }
        return this;
    }

    /**
     * 根据字符串同时设置主机和端口号
     * 
     * @param hostPortStr
     *            格式为<i>host[:port]</i>
     */
    public void setHostAndPort(String hostPortStr)
    {
        if (StringUtils.isBlank(hostPortStr))
        {
            throw new IllegalArgumentException("hostPortStr");
        }
        if (hostPortStr.indexOf(":") > 0)
        {
            this.setHost(hostPortStr.substring(0, hostPortStr.indexOf(":")));
            this.setPort(Integer.parseInt(hostPortStr.substring(hostPortStr
                    .indexOf(":") + 1)));
        }
        else
        {
            this.setHost(hostPortStr);
            this.setPort(80);
        }
    }

    public HttpUrl copy()
    {
        HttpUrl holder = new HttpUrl(host, port, requestUri, paramsEncoding);
        for (Entry<String, String[]> en : params.entrySet())
        {
            String[] arr = new String[en.getValue().length];
            System.arraycopy(en.getValue(), 0, arr, 0, en.getValue().length);
            holder.params.put(en.getKey(), arr);
        }
        return holder;
    }

    @Override
    public String toString()
    {
        StringBuilder builder = new StringBuilder();
        if (isHttps)
        {
            builder.append("https://");
        }
        else
        {
            builder.append("http://");
        }
        builder.append(host);
        if (isHttps)
        {
            if (port != 443)
            {
                builder.append(":").append(port);
            }
        }
        else
        {
            if (port != 80)
            {
                builder.append(":").append(port);
            }
        }
        if (!requestUri.startsWith("/"))
        {
            builder.append("/");
        }
        builder.append(requestUri);
        String paramStr = null;
        try
        {
            paramStr = params.toString(true);
        }
        catch (UnsupportedEncodingException e)
        {
        }
        if (StringUtils.isNotEmpty(paramStr))
        {
            builder.append("?").append(paramStr);
        }
        return builder.toString();
    }

    /**
     * 获取当前请求的URL路径对象
     * 
     * @param req
     * @param ignoreKeys
     *            需要忽略的参数key
     * @return
     */
    public static HttpUrl fromHttpRequest(HttpServletRequest req,
            String... ignoreKeys)
    {
        String pageUri = req.getRequestURI();
        if (pageUri == null)
        {
            pageUri = "/";
        }
        HttpUrl holder = new HttpUrl(req.getServerName(), req.getServerPort(),
                pageUri);
        @SuppressWarnings("unchecked")
        Map<String, String[]> params = req.getParameterMap();
        if (params == null)
        {
            return holder;
        }
        for (Entry<String, String[]> en : params.entrySet())
        {
            if (ignoreKeys != null)
            {
                boolean ignore = false;
                for (String key : ignoreKeys)
                {
                    if (en.getKey().equals(key))
                    {
                        ignore = true;
                        break;
                    }
                }
                if (ignore)
                {
                    continue;
                }
            }
            holder.getParams().put(en.getKey(), en.getValue());
        }
        return holder;
    }

    /**
     * 从url串中解析出对象
     * 
     * @param url
     * @param paramEncoding
     * @return
     */
    public static HttpUrl fromUrl(String url, String paramEncoding)
    {
        URL u = null;
        try
        {
            u = new URL(url);
        }
        catch (MalformedURLException e)
        {
            throw new IllegalArgumentException("illegal url string:" + url, e);
        }
        String protocol = u.getProtocol();
        if (!("http".equalsIgnoreCase(u.getProtocol()) || "https"
                .equalsIgnoreCase(protocol)))
        {
            throw new IllegalArgumentException("illegal http protocol:"
                    + protocol);
        }
        HttpUrl holder = new HttpUrl();
        holder.setHttps("https".equalsIgnoreCase(protocol));
        holder.setHost(u.getHost());
        holder.setPort(u.getPort() == -1 ? u.getDefaultPort() : u.getPort());
        holder.setRequestUri(u.getPath());
        if (StringUtils.isNotEmpty(paramEncoding))
        {
            holder.setParamsEncoding(paramEncoding);
        }
        try
        {
            holder.getParams().addParamFromString(u.getQuery(), paramEncoding);
        }
        catch (UnsupportedEncodingException e)
        {
            throw new IllegalArgumentException(
                    "unsupported url param encoding:" + paramEncoding, e);
        }
        return holder;
    }

    /**
     * 代表一个有序的URL参数对象
     * 
     * @author zengjie
     * @since 2013-9-27
     * @see LinkedHashMap
     * @see HttpUrl#addOrReplace(String, Object...)
     * @see HttpUrl#setParam(String, Object, boolean)
     * @see HttpUrl#toString(boolean)
     */
    public class Params extends LinkedHashMap<String, String[]>
    {

        private static final long serialVersionUID = -890157917077460545L;

        Params()
        {
        }

        private Params(int capacity)
        {
            super(capacity);
        }

        /**
         * 设置参数
         * 
         * @param key
         * @param value
         * @param replace
         *            是否覆盖已经存在指定key的参数
         * @return
         */

        /**
         * 从字符串中解析出参数
         * 
         * @param qString
         * @throws UnsupportedEncodingException
         */
        void addParamFromString(String qString, String encoding)
                throws UnsupportedEncodingException
        {
            if (StringUtils.isBlank(qString))
            {
                return;
            }
            qString = qString.trim();
            String[] pairs = qString.split("&");
            if (ArrayUtils.isEmpty(pairs))
            {
                return;
            }
            for (String param : pairs)
            {
                String[] arr = param.split("=", 2);
                if (ArrayUtils.isEmpty(arr) || arr.length != 2)
                {
                    continue;
                }
                setParam(
                        StringUtils.isEmpty(encoding) ? arr[0]
                                : URLDecoder.decode(arr[0], paramsEncoding),
                        StringUtils.isEmpty(encoding) ? arr[1] : URLDecoder
                                .decode(arr[1], paramsEncoding), false);
            }
        }

        /**
         * 将参数转换成URL字符串表现形式
         * 
         * @param encode
         *            是否对参数用HttpUrl指定的paramEncoding进行编码
         * @return
         * @throws UnsupportedEncodingException
         */
        public String toString(boolean encode)
                throws UnsupportedEncodingException
        {
            StringBuilder builder = new StringBuilder();
            for (java.util.Map.Entry<String, String[]> en : entrySet())
            {
                String key = encode ? URLEncoder.encode(en.getKey(),
                        paramsEncoding) : en.getKey();
                for (String value : en.getValue())
                {
                    builder.append(key);
                    builder.append("=");
                    if (encode)
                    {
                        builder.append(URLEncoder.encode(value, paramsEncoding));
                    }
                    else
                    {
                        builder.append(value);
                    }
                    builder.append("&");
                }
            }
            if (builder.length() > 0)
            {
                builder.delete(builder.length() - 1, builder.length());
            }
            return builder.toString();
        }

        @Override
        public Object clone()
        {
            Params params = new Params(this.size());
            String[] arr;
            for (java.util.Map.Entry<String, String[]> en : entrySet())
            {
                arr = new String[en.getValue().length];
                System.arraycopy(en.getValue(), 0, arr, 0, en.getValue().length);
                params.put(en.getKey(), arr);
            }
            return params;
        }

        @Override
        public String toString()
        {
            try
            {
                return toString(false);
            }
            catch (UnsupportedEncodingException e)
            {
                return null;
            }
        }
    }
}
