package cn.pconline.search.common.log;

import java.io.BufferedOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.NoSuchElementException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;

/**
 * 日志输入输出(在IO层对日志读写进行了一个封装)
 * 
 * @author zengjie
 * @since 2013-9-23
 * @see
 */
public class LogIO implements Closeable
{

    private static final String LOG_FILE_PREFIX = "SearchLog.";

    private static final Charset CHARSET = Charset.forName("GBK");

    // private boolean enableCompress;

    private Map<String, OutWrapper> outMap = new ConcurrentHashMap<String, LogIO.OutWrapper>();

    private Object writeLock = new Object();

    private String logFileFloder;

    public LogIO(String logFileFloder)
    {
        this.logFileFloder = logFileFloder;
    }

    public LogIO()
    {
    }

    /**
     * 向本地系统写入一条日志
     * 
     * @param log
     * @throws IOException
     */
    public void writeLog(SearchLog log) throws IOException
    {
        OutWrapper wrapper = null;
        try
        {
            wrapper = getOut(log);
            byte[] content = log.getKey().getBytes(CHARSET);
            byte[] app = log.getApp().getBytes(CHARSET);
            ByteBuffer buffer = ByteBuffer.allocate(content.length + 1
                    + app.length + 1 + 4 + 8);
            buffer.put((byte) app.length);
            buffer.put(app);
            buffer.put((byte) content.length);
            buffer.put(content);
            buffer.putInt(log.getRetCount());
            buffer.putLong(log.getSearchTime().getTime());

            synchronized (writeLock)
            {
                wrapper.out.write(buffer.array());
            }
        }
        finally
        {
            releaseOut(wrapper);
        }
    }

    /**
     * 获取昨天搜索日志的一个迭代器,使用完该迭代器需调用 {@link LogIterator#close()}关闭
     * 
     * @return
     * @throws IOException
     */
    public LogIterator getYesterdayLogs() throws IOException
    {
        File f = null;
        if (StringUtils.isNotEmpty(logFileFloder))
        {
            f = new File(logFileFloder, LOG_FILE_PREFIX
                    + getFileKey(getYesterday()));
        }
        else
        {
            f = new File(LOG_FILE_PREFIX + getFileKey(getYesterday()));
        }
        if (f.exists() && f.isFile() && f.length() > 0)
        {
            final InputStream in = new FileInputStream(f);
            return new LogIterator(in, new Iterator<SearchLog>()
            {

                private SearchLog log;

                private boolean EOF = false;

                @Override
                public void remove()
                {
                    throw new UnsupportedOperationException();
                }

                @Override
                public SearchLog next()
                {
                    if (EOF)
                    {
                        throw new NoSuchElementException();
                    }
                    if (log == null)
                    {
                        if (!hasNext())
                        {
                            throw new NoSuchElementException();
                        }
                    }
                    SearchLog ret = log;
                    log = null;
                    return ret;
                }

                @Override
                public boolean hasNext()
                {
                    if (EOF)
                    {
                        return false;
                    }
                    try
                    {
                        log = read(in);
                        if (log == null)
                        {
                            EOF = true;
                            return false;
                        }
                        return true;
                    }
                    catch (IOException e)
                    {
                        EOF = true;
                        return false;
                    }
                }
            });
        }
        return null;
    }

    private SearchLog read(InputStream in) throws IOException
    {
        // int count = 0;
        SearchLog log = new SearchLog();
        if (in.available() <= 0)
        {
            return null;
        }
        byte appLength = (byte) in.read();
        if (appLength <= 0)
        {
            return null;
        }
        byte[] app = new byte[appLength];
        if (in.available() <= 0 || in.read(app) < appLength)
        {
            return null;
        }
        log.setApp(new String(app, CHARSET));

        if (in.available() <= 0)
        {
            return null;
        }
        byte contentLength = (byte) in.read();
        if (contentLength <= 0)
        {
            return null;
        }
        byte[] content = new byte[contentLength];
        if (in.available() <= 0 || in.read(content) < contentLength)
        {
            return null;
        }
        log.setKey(new String(content, CHARSET));

        byte[] arr = new byte[12];
        if (in.available() <= 0 || in.read(arr) < arr.length)
        {
            return null;
        }
        ByteBuffer buffer = ByteBuffer.wrap(arr);
        log.setRetCount(buffer.getInt());
        log.setSearchTime(new Date(buffer.getLong()));
        return log;
    }

    private static Date getToday()
    {
        Calendar c = Calendar.getInstance();
        c.set(Calendar.HOUR, 0);
        c.set(Calendar.MINUTE, 0);
        c.set(Calendar.SECOND, 0);
        c.set(Calendar.MILLISECOND, 0);
        return c.getTime();
    }

    public static Date getYesterday()
    {
        Calendar c = Calendar.getInstance();
        c.add(Calendar.DATE, -1);
        c.set(Calendar.HOUR, 0);
        c.set(Calendar.MINUTE, 0);
        c.set(Calendar.SECOND, 0);
        c.set(Calendar.MILLISECOND, 0);
        return c.getTime();
    }

    public static String getFileKey(Date date)
    {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
        String key = sdf.format(date);
        return key;
    }

    /**
     * 获取输出流
     * 
     * @param log
     * @return
     */
    private OutWrapper getOut(SearchLog log) throws IOException
    {
        String key = getFileKey(log.getSearchTime());
        OutWrapper wrapper = outMap.get(key);
        if (wrapper == null)
        {
            synchronized (outMap)
            {
                wrapper = outMap.get(key);
                if (wrapper == null)
                {
                    String yesterdayKey = getFileKey(getYesterday());
                    OutWrapper yesterday = outMap.get(yesterdayKey);
                    if (yesterday != null)
                    {
                        IOUtils.closeQuietly(yesterday.out);
                        outMap.remove(yesterdayKey);
                    }

                    wrapper = new OutWrapper();
                    wrapper.file = StringUtils.isEmpty(logFileFloder) ? new File(
                            LOG_FILE_PREFIX + key) : new File(logFileFloder,
                            LOG_FILE_PREFIX + key);
                    createFolder(wrapper.file.getParentFile());
                    wrapper.out = new FileOutputStream(wrapper.file, true);
                    wrapper.out = new BufferedOutputStream(wrapper.out,
                            1024 * 512);// 内存缓冲
                    wrapper.dateFor = getToday();
                    outMap.put(key, wrapper);
                }
            }
        }
        wrapper.refCount.incrementAndGet();
        return wrapper;
    }

    private void createFolder(File file) throws IOException
    {
        if (file == null || file.exists())
        {
            return;
        }
        createFolder(file.getParentFile());
        if (!file.mkdir())
        {
            throw new IOException("create folder [" + file + "] fail");
        }
    }

    private void releaseOut(OutWrapper wrapper)
    {
        if (wrapper == null)
        {
            return;
        }
        wrapper.refCount.decrementAndGet();
        if (wrapper.refCount.get() == 0
                && wrapper.dateFor.getTime() < getToday().getTime())
        {
            synchronized (outMap)
            {
                IOUtils.closeQuietly(wrapper.out);
                outMap.remove(getFileKey(wrapper.dateFor));
            }
        }
    }

    private class OutWrapper
    {

        private Date dateFor;

        private File file;

        private OutputStream out;

        private AtomicLong refCount = new AtomicLong(0);

    }

    @Override
    public void close() throws IOException
    {
        synchronized (outMap)
        {
            Iterator<Entry<String, OutWrapper>> it = outMap.entrySet()
                    .iterator();
            while (it.hasNext())
            {
                IOUtils.closeQuietly(it.next().getValue().out);
                it.remove();
            }
        }
    }
}
