package cn.pconline.search.common.util;

import java.lang.Character.UnicodeBlock;
import java.math.BigDecimal;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang.StringUtils;

public class QueryUtil
{

    /**
     * 根据指定关键字和权值Map转换成solr内嵌edismax查询
     * 
     * @param keyword
     * @param boostMap
     * @return
     */
    public static String toDismaxQuery(String keyword,
            Map<String, Float> boostMap)
    {
        StringBuilder builder = new StringBuilder();
        builder.append("_query_:\"{!edismax qf='");
        for (Entry<String, Float> en : boostMap.entrySet())
        {
            builder.append(en.getKey()).append("^")
                    .append(new BigDecimal(en.getValue()).toPlainString())
                    .append(" ");
        }
        builder.append("'}").append(escapeQStr(keyword));
        return builder.toString();
    }

    private static Map<Pattern, String> escapeMap = new LinkedHashMap<Pattern, String>();

    private static void addToEscapeMap(String key, String value)
    {
        escapeMap.put(Pattern.compile(key, Pattern.LITERAL),
                Matcher.quoteReplacement(value));
    }

    static
    {
        // 一般转义
        addToEscapeMap("\\", "\\\\");
        addToEscapeMap("\"", "\\\"");
        addToEscapeMap("/", "\\/");
        // LUCENE特殊字符
        addToEscapeMap("?", "\\?");
        addToEscapeMap("*", "\\*");
        addToEscapeMap(":", "\\:");
        addToEscapeMap("~", "\\~");
        addToEscapeMap("+", "\\+");
        addToEscapeMap("-", "\\-");
        addToEscapeMap("^", "\\^");
        // 各种括号
        addToEscapeMap("(", "\\(");
        addToEscapeMap(")", "\\)");
        addToEscapeMap("[", "\\[");
        addToEscapeMap("]", "\\]");
        addToEscapeMap("{", "\\{");
        addToEscapeMap("}", "\\}");
        // LUCENE查询连接符
        addToEscapeMap("OR", "\\OR");
        addToEscapeMap("AND", "\\AND");
        addToEscapeMap("NOT", "\\NOT");
    }

    /**
     * 转义查询中的特殊字符串
     * 
     * @param query
     * @return
     */
    public static String escapeQStr(String query)
    {
        if (StringUtils.isEmpty(query))
        {
            return query;
        }
        query = query.trim();
        for (Entry<Pattern, String> e : escapeMap.entrySet())
        {
            query = e.getKey().matcher(query).replaceAll(e.getValue());
        }
        return query;
    }

    /**
     * 判断一个字符串是否使用PharsQuery查询
     * 
     * @param input
     * @return
     */
    public static boolean shouldUsePharse(String input)
    {
        if (StringUtils.isBlank(input))
        {
            return false;
        }
        input = input.trim();
        for (int i = 0; i < input.length(); i++)
        {
            char c = input.charAt(i);
            if (Character.isSpaceChar(input.charAt(i))
                    || Character.isWhitespace(input.charAt(i)))
            {
                return false;
            }
            if (!Character.isDigit(input.charAt(i))
                    && !Character.isUpperCase(c) && !Character.isLowerCase(c)
                    && input.charAt(i) != '_' && input.charAt(i) != '-')
            {
                return false;
            }
        }
        return true;
    }

    /**
     * 对查询串根据一定的规则进行截断,符合要求长度
     * 
     * @param q
     * @param max
     *            最大允许中文字数，（1中文=2英文或数字）
     * @return
     */
    public static String subQueryString(String q, int max)
    {
        if (q.length() <= max)
        {
            return q;
        }
        max *= 2;
        int blen = getStringBLen(q);
        if (blen <= max)
        {
            return q;
        }
        q = q.trim();
        StringBuilder out = new StringBuilder(q);
        do
        {
            char last = out.charAt(out.length() - 1);
            out.delete(out.length() - 1, out.length());
            UnicodeBlock ub = UnicodeBlock.of(last);
            if (ub == UnicodeBlock.ARABIC)
            {
                while (true)
                {
                    if (out.length() == 0)
                    {
                        break;
                    }
                    ub = UnicodeBlock.of(out.charAt(out.length() - 1));
                    if (ub == UnicodeBlock.ARABIC)
                    {
                        out.delete(out.length() - 1, out.length());
                        continue;
                    }
                    break;
                }
            }
            else if (isLetter(last))
            {
                while (true)
                {
                    if (out.length() == 0)
                    {
                        break;
                    }
                    if (isLetter(out.charAt(out.length() - 1)))
                    {
                        out.delete(out.length() - 1, out.length());
                        continue;
                    }
                    break;
                }
            }
        }
        while (out.length() > 0 && getStringBLen(out.toString()) > max);
        if (out.length() == 0)
        {
            return q.substring(0, max);
        }
        return out.toString();
    }

    private static boolean isLetter(char c)
    {
        if ((c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z'))
        {
            return true;
        }
        return false;
    }

    private static int getStringBLen(String input)
    {
        if (StringUtils.isEmpty(input))
        {
            return 0;
        }
        int len = 0;
        UnicodeBlock ub = null;
        for (int i = 0; i < input.length(); i++)
        {
            char c = input.charAt(i);
            ub = UnicodeBlock.of(c);
            if (ub == Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS
                    || ub == Character.UnicodeBlock.CJK_COMPATIBILITY_IDEOGRAPHS
                    || ub == Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS_EXTENSION_A)
            {
                len += 2;
            }
            else
            {
                len += 1;
            }
        }
        return len;
    }
}
