package cn.pconline.search.common.log;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;

import cn.pconline.search.common.Configuration;
import cn.pconline.search.common.log.GroupCounter.Item;
import cn.pconline.search.common.util.DailyFixTimeRunThread;

/**
 * 日志保存线程，该线程每运行一次将会把昨天的搜索日志进行入库操作
 * 
 * @author zengjie
 * @since 2013-9-23
 * @see
 */
public class LogSaveThread extends DailyFixTimeRunThread
{

    private static Logger logger = Logger.getLogger(LogSaveThread.class);

    private static final int WEEK = 1; // 一周的开始,或者本周还没有进行过更新

    private static final int MONTH = 2; // 一个月的开始,或者本月还没有进行过更新

    private static final int YEAR = 3; // 一年的开始,或者本年还没有进行过更新

    private static final int WEEK_MONTH = 4;// 周、月的开始

    private static final int WEEK_YEAR = 5;// 周、年的开始

    private static final int MONTH_YEAR = 6;// 月、年的开始

    private static final int WEEK_MONTH_YEAR = 7;// 周、月、年的开始

    private LogIO logIO;

    private Configuration config;

    private Map<SearchLog, Integer> keyOccurMap;

    /**
     * constructor
     * 
     * @param logIO
     *            日志输入输出对象
     * @param config
     *            相关配置
     * @param runTime
     *            保存日志运行时间点,在每天的指定时间点(通过runTime的hour、minue、second，
     *            millionsecond来指定)
     */
    public LogSaveThread(LogIO logIO, Configuration config)
    {
        super(config.getConfig("searchLogSaveTime"), "SearchLog_Save_Thread");
        this.logIO = logIO;
        this.config = config;
        testDb();
    }

    private void testDb()
    {
        logger.debug("Testing search log db connection...");
        Connection conn = null;
        try
        {
            conn = getConnection();
        }
        catch (Exception e)
        {
            throw new RuntimeException(
                    "Error while try establish db connection", e);
        }
        finally
        {
            if (conn != null)
            {
                try
                {
                    conn.close();
                }
                catch (SQLException e)
                {
                }
            }
        }
    }

    private Connection getConnection() throws SQLException
    {
        String adDbUrl = config.require("ad.dburl");
        String user = config.require("ad.user");
        String passwd = config.require("ad.passwd");
        DriverManager.registerDriver(new oracle.jdbc.OracleDriver());
        return DriverManager.getConnection(adDbUrl, user, passwd);
    }

    @Override
    protected void doRun()
    {
        if (Thread.currentThread().isInterrupted())
        {
            return;
        }
        keyOccurMap = new HashMap<SearchLog, Integer>();
        long startTime = System.currentTimeMillis();

        Connection con = null;
        try
        {
            con = getConnection();
            con.setAutoCommit(false);

            int count = getUpdateOrSave(con, Integer.MAX_VALUE, getNowTime());// 读取本地统计文件

            con.commit();
            long endTime = System.currentTimeMillis();
            logger.info("关键字入库完毕!所用时间: " + (endTime - startTime) + ",总数据条数:"
                    + count);
        }
        catch (Exception e)
        {
            logger.error("Save search log error", e);
        }
        finally
        {
            try
            {
                if (con != null) con.close();
            }
            catch (SQLException e)
            {
                logger.error("", e);
            }
        }
    }

    protected int getUpdateOrSave(Connection conn, int topNum, int time)
            throws Exception
    {
        ResultSet rs = null;
        PreparedStatement existSql = null;
        PreparedStatement updatePs = null;
        PreparedStatement insertPs = null;
        int totalCount = 0;
        try
        {
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd");
            boolean hasUpdate = false;

            List<Item> list = SearchLogHelper.getTop(logIO,
                    topNum < 0 ? Integer.MAX_VALUE : topNum, keyOccurMap);

            for (int i = 0, c = list.size(); i < c; ++i)
            {
                if (Thread.currentThread().isInterrupted())
                {
                    throw new InterruptedException();
                }
                Item item = list.get(i);
                int count = item.getCount();
                Integer res = keyOccurMap.get(item.getKey());
                if (existSql == null)
                {
                    existSql = conn
                            .prepareStatement("select updatetime,week_num,month_num,yesterday_num,year_num from SEARCHKEYLOG sl where sl.keyword=? and sl.app=?");
                }
                existSql.setString(1, item.getKey().getKey());
                existSql.setString(2, item.getKey().getApp());
                rs = existSql.executeQuery();
                if (rs.next())// 已经存在记录,对值进行更新
                {
                    hasUpdate = sdf.format(rs.getDate("updatetime")).equals(
                            LogIO.getFileKey(LogIO.getYesterday()));
                    if (updatePs == null)
                    {
                        updatePs = conn
                                .prepareStatement("update KS_SEARCHKEY_LOG sl SET sl.summary=sl.summary+?,sl.week_num=?,sl.month_num=?,sl.yesterday_num=?"
                                        + ",sl.last_search_num=?,sl.year_num=?,sl.updatetime=sysdate-1 where sl.keyword=? and sl.app= ?");
                    }
                    updatePs.setInt(1, count);
                    updatePs.setInt(
                            2,
                            time == WEEK || time == WEEK_YEAR
                                    || time == WEEK_MONTH
                                    || time == WEEK_MONTH_YEAR ? count : rs
                                    .getInt("week_num") + count);
                    updatePs.setInt(
                            3,
                            time == MONTH || time == MONTH_YEAR
                                    || time == WEEK_MONTH
                                    || time == WEEK_MONTH_YEAR ? count : rs
                                    .getInt("month_num") + count);
                    updatePs.setInt(4, hasUpdate ? rs.getInt("yesterday_num")
                            + count : count);
                    updatePs.setInt(5, res);
                    updatePs.setInt(
                            6,
                            time == YEAR || time == MONTH_YEAR
                                    || time == WEEK_YEAR
                                    || time == WEEK_MONTH_YEAR ? count : rs
                                    .getInt("year_num") + count);
                    updatePs.setString(7, item.getKey().getKey());
                    updatePs.setString(8, item.getKey().getApp());
                    updatePs.addBatch();
                    hasUpdate = false;
                }
                else
                {
                    // 重新插入数据
                    if (insertPs == null)
                    {
                        insertPs = conn
                                .prepareStatement("insert into KS_SEARCHKEY_LOG(id, keyword, app, summary, createtime, updatetime, week_num, month_num, yesterday_num, year_num, last_search_num)"
                                        + " values(SEQ_SERARCHKEY_ID.nextval,?,?,?,sysdate-1,sysdate-1,?,?,?,?,?)");
                    }
                    insertPs.setString(1, item.getKey().getKey());
                    insertPs.setString(2, item.getKey().getApp());
                    insertPs.setInt(3, count);
                    insertPs.setInt(4, count);
                    insertPs.setInt(5, count);
                    insertPs.setInt(6, count);
                    insertPs.setInt(7, count);
                    insertPs.setInt(8, res);
                    insertPs.addBatch();
                }
                totalCount++;
                rs.close();
            }
            if (updatePs != null)
            {
                updatePs.executeBatch();
            }
            if (insertPs != null)
            {
                insertPs.executeBatch();
            }
        }
        finally
        {
            if (rs != null)
            {
                try
                {
                    rs.close();
                }
                catch (SQLException e)
                {
                }
            }
            closeState(updatePs);
            closeState(insertPs);
            closeState(existSql);
        }
        return totalCount;
    }

    private static void closeState(Statement s)
    {
        if (s != null)
        {
            try
            {
                s.close();
            }
            catch (SQLException e)
            {
            }
        }
    }

    /**
     * 得到当前的前一天日期性质: 一周的开始,或者本周还没有进行过更新, 一个月的开始,或者本月还没有进行过更新, 一年的开始,或者本年还没有进行过更新
     * 
     * @return
     */
    private static int getNowTime()
    {
        Calendar c = Calendar.getInstance();
        c.set(Calendar.DATE, c.get(Calendar.DATE) - 1);// 昨天

        boolean isWeek = c.get(Calendar.DAY_OF_WEEK) - 1 == 1;// 今日星期几(星期一记起为第一天)
        boolean isMonth = c.get(Calendar.DATE) == c
                .getMinimum(Calendar.DAY_OF_MONTH);// 本月的第一天是几号
        boolean isYear = c.get(Calendar.DAY_OF_YEAR) == 1;// 今天是今年的第几天

        if (isWeek && isMonth && isYear) return WEEK_MONTH_YEAR;
        if (isWeek && isMonth) return WEEK_MONTH;
        if (isWeek && isYear) return WEEK_YEAR;
        if (isMonth && isYear) return MONTH_YEAR;
        if (isWeek) return WEEK;
        if (isMonth) return MONTH;
        if (isYear) return YEAR;

        return -1;
    }

}
