package cn.pconline.search.common;

import java.io.IOException;
import java.io.InputStream;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TimeZone;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.http.HttpHost;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.Credentials;
import org.apache.http.auth.NTCredentials;
import org.apache.http.client.HttpClient;
import org.apache.http.conn.params.ConnRoutePNames;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.impl.BinaryResponseParser;
import org.apache.solr.client.solrj.impl.HttpSolrServer;
import org.apache.solr.client.solrj.request.QueryRequest;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.apache.solr.common.SolrException;
import org.apache.solr.common.util.FastInputStream;
import org.apache.solr.common.util.JavaBinCodec;
import org.apache.solr.common.util.NamedList;

import cn.pconline.search.common.ks.KSResult;
import cn.pconline.search.common.query.SolrQueryResult;
import cn.pconline.search.common.util.HttpUrl;
import cn.pconline.search.common.util.Intf;

import com.alibaba.fastjson.JSON;

/**
 * solr搜索器 <br/>
 * <i><b>NOTE:</b>本类不仅限于对solr服务器进行搜索,还包括对旧快搜的search API调用,详见:
 * {@link #searchKs(HttpUrl)}</i>
 * 
 * @author zengjie
 * @since 2013-9-16
 * @see
 */
public class SolrSearcher
{

    public static final String DEFAULT_SOLR_URL = "http://192.168.239.79:8983/solr/";

    private SolrServer server;

    private Configuration config;

    /**
     * constructor
     * 
     * @param solrBaseUrl
     * @param uniqueName
     *            用来缓存Searcher的全局唯一key
     */
    public SolrSearcher(String solrBaseUrl, Configuration config)
    {
        HttpSolrServer httpServer = new HttpSolrServer(solrBaseUrl, null,
                new TimeZoneParser());
        this.config = config;
        setupHttpServer(httpServer);
        this.server = httpServer;
    }

    public SolrSearcher(Configuration config)
    {
        this(config.getConfig("searchServer", DEFAULT_SOLR_URL), config);
    }

    /**
     * 执行搜索操作,返回solr的查询响应,默认使用select handler
     * 
     * @param index
     * @param q
     * @return
     */
    public QueryResponse search(String index, SolrQuery q)
    {
        return search(index, "select", q);
    }

    /**
     * 查询返回封装结果
     * 
     * @param index
     * @param q
     * @return
     */
    public SolrQueryResult searchAs(String index, SolrQuery q)
    {
        return SolrQueryResult.fromResponse(search(index, "select", q));
    }

    /**
     * 使用指定的handler搜索指定的index
     * 
     * @param index
     * @param handler
     * @param q
     * @return
     */
    public QueryResponse search(String index, String handler, SolrQuery q)
    {
        QueryRequest req = new QueryRequest(q);
        req.setPath("/" + index + "/" + handler);
        try
        {
            return req.process(server);
        }
        catch (SolrServerException e)
        {
            throw new SearchException(e);
        }
    }

    /**
     * 对旧快搜进行搜索并且返回结果
     * <p>
     * 本方法会覆盖设置的返回类型参数,统一使用JSON通信
     * </p>
     * 
     * @param searchUrl
     * @return 快搜搜索结果对象
     * @throws IOException
     */
    public KSResult searchKs(HttpUrl searchUrl) throws IOException
    {
        searchUrl.replaceParam("returnType", "json");
        String rsp = Intf.readHtml4Get(searchUrl.toString(), null, "gbk", 3000,
                2);
        if (StringUtils.isBlank(rsp))
        {
            return new KSResult();
        }
        return JSON.parseObject(rsp.trim(), KSResult.class);
    }

    /**
     * 对HTTP Server的属性进行相关设置
     * 
     * @param server
     */
    protected void setupHttpServer(HttpSolrServer server)
    {
        server.setConnectionTimeout(config.getIntConfig("http.ConnectTimeout",
                5000));
        server.setSoTimeout(config.getIntConfig("http.SoTimeout", 5000));
        server.setMaxTotalConnections(config.getIntConfig(
                "http.ConnectionCount", 100));
        server.setMaxRetries(config.getIntConfig("http.retryTimes", 2));
        // 设置是否使用代理
        if (config.getBooleanConfig("http.useProxy", false))
        {
            HttpClient client = server.getHttpClient();
            HttpHost proxy = new HttpHost(config.getConfig("http.proxyIp"),
                    config.getIntConfig("http.proxyPort"));
            client.getParams().setParameter(ConnRoutePNames.DEFAULT_PROXY,
                    proxy);
            if (client instanceof DefaultHttpClient)
            {
                String user = config.getConfig("http.proxyUser");
                String pwd = config.getConfig("http.proxyPassword");
                if (StringUtils.isNotEmpty(user) && StringUtils.isNotEmpty(pwd))
                {
                    Credentials credentials = new NTCredentials(user, pwd, "",
                            "");
                    ((DefaultHttpClient) client).getCredentialsProvider()
                            .setCredentials(AuthScope.ANY, credentials);
                }
            }
        }
    }

    /**
     * 对响应返回的高亮结果和原始字段进行document合并
     * <p>
     * solr返回的document和highlight結果是是分开的两部分,此方法将高亮结果返回的字段合并至document中,
     * 如果参数resultMaps传了,则将document和highlight结果合并到一個新的Map中并且放到resultMaps中
     * </p>
     * 
     * @param rsp
     * @param keyField
     *            主键字段的名称
     * @param resultMaps
     *            如果此参数为空，则直接将高亮直接合并至原始document,否则新生成一个Map
     */
    public static void mergeHighLight(QueryResponse rsp, String keyField,
            List<Map<String, Object>> resultMaps)
    {
        SolrDocumentList rspList = rsp.getResults();
        if (CollectionUtils.isEmpty(rspList)
                || MapUtils.isEmpty(rsp.getHighlighting()))
        {
            return;
        }
        Object key = null;
        Map<String, Object> result = null;
        boolean useSource = resultMaps == null;
        for (SolrDocument doc : rspList)
        {
            if (useSource)
            {
                result = doc;
            }
            else
            {
                result = new HashMap<String, Object>();
            }
            key = doc.get(keyField);
            Map<String, List<String>> map = rsp.getHighlighting().get(key);
            if (MapUtils.isNotEmpty(map))
            {
                if (!useSource)
                {
                    result.putAll(doc);
                }
                for (Entry<String, List<String>> en : map.entrySet())
                {
                    if (en.getValue().size() == 1)
                    {
                        result.put(en.getKey(), en.getValue().get(0));
                    }
                    else
                    {
                        result.put(en.getKey(), en.getValue());
                    }
                }
            }
            if (!useSource)
            {
                resultMaps.add(result);
            }
        }
    }

    /**
     * 关闭当前搜索器
     */
    void close()
    {
        server.shutdown();
    }

    private static class TimeZoneParser extends BinaryResponseParser
    {

        private static final TimeZone CURRENT_TIME_ZONE = TimeZone.getDefault();

        private static final long OFFSET_TZ = CURRENT_TIME_ZONE
                .getOffset(System.currentTimeMillis());

        @SuppressWarnings("unchecked")
        public NamedList<Object> processResponse(InputStream body,
                String encoding)
        {
            try
            {
                return (NamedList<Object>) new JavaBinCodec()
                {

                    public Object readVal(FastInputStream dis)
                            throws IOException
                    {
                        tagByte = dis.readByte();
                        switch (tagByte >>> 5)
                        {
                        case STR >>> 5:
                            return readStr(dis);
                        case SINT >>> 5:
                            return readSmallInt(dis);
                        case SLONG >>> 5:
                            return readSmallLong(dis);
                        case ARR >>> 5:
                            return readArray(dis);
                        case ORDERED_MAP >>> 5:
                            return readOrderedMap(dis);
                        case NAMED_LST >>> 5:
                            return readNamedList(dis);
                        case EXTERN_STRING >>> 5:
                            return readExternString(dis);
                        }

                        switch (tagByte)
                        {
                        case NULL:
                            return null;
                        case DATE:
                            // TODO 由于SOLR处理时间都是当做UTC时间去处理和序
                            // 列化的 所以在读取solr返回的时间的时候我们需要把
                            // solr转换成的UTC时间减去当前时区和UTC时区的偏移量
                            return new Date(dis.readLong() - OFFSET_TZ);
                        case INT:
                            return dis.readInt();
                        case BOOL_TRUE:
                            return Boolean.TRUE;
                        case BOOL_FALSE:
                            return Boolean.FALSE;
                        case FLOAT:
                            return dis.readFloat();
                        case DOUBLE:
                            return dis.readDouble();
                        case LONG:
                            return dis.readLong();
                        case BYTE:
                            return dis.readByte();
                        case SHORT:
                            return dis.readShort();
                        case MAP:
                            return readMap(dis);
                        case SOLRDOC:
                            return readSolrDocument(dis);
                        case SOLRDOCLST:
                            return readSolrDocumentList(dis);
                        case BYTEARR:
                            return readByteArray(dis);
                        case ITERATOR:
                            return readIterator(dis);
                        case END:
                            return END_OBJ;
                        case SOLRINPUTDOC:
                            return readSolrInputDocument(dis);
                        }

                        throw new RuntimeException("Unknown type " + tagByte);
                    }
                }.unmarshal(body);
            }
            catch (IOException e)
            {
                throw new SolrException(SolrException.ErrorCode.SERVER_ERROR,
                        "parsing error", e);

            }
        }
    }
}
