package cn.pconline.search.common.tools.sensitive;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import cn.pconline.search.common.Configuration;
import cn.pconline.search.common.util.DailyFixTimeRunThread;

/**
 * 基于对输入进行分析匹配的敏感词过滤器
 * 
 * @author zengjie
 * @since 2013-10-9
 * @see
 */
public class AnalyzeSensitiveFilter extends DailyFixTimeRunThread implements
        SensitiveFilter
{

    private static final Logger logger = Logger
            .getLogger(AnalyzeSensitiveFilter.class);

    private ReentrantLock loadLock = new ReentrantLock();

    private String wordDb;

    protected volatile WordMapping mapping = null;

    private String domain;

    private volatile boolean closed = false;

    public AnalyzeSensitiveFilter(Configuration config)
    {
        super(config.getConfig("sensitiveWordReloadTime", ""),
                "SensitiveWord-Load-Thread");
        wordDb = config.require("sensitiveWordDb");
        domain = config.getConfig("sensitiveFilterDomain", "pconline");
        doLoad();
        this.start();
    }

    @Override
    protected void doRun()
    {
        doLoad();
    }

    @Override
    public SensitiveMatcher matchSensitive(String input)
    {
        return hasSensitiveInMapping(input, mapping);
    }

    private SensitiveMatcher hasSensitiveInMapping(String input,
            WordMapping mapping)
    {
        if (StringUtils.isBlank(input))
        {
            return SensitiveMatcher.EMPTY_MATCHER;
        }
        input = input.trim();
        if (input.length() < mapping.getMinWordLen())
        {
            return SensitiveMatcher.EMPTY_MATCHER;
        }
        if (!mapping.hasMapping())
        {
            return SensitiveMatcher.EMPTY_MATCHER;
        }
        return new SensitiveMatcherImpl(mapping, input);
    }

    @Override
    public void reload(SensitiveWordSource source)
    {
        loadLock.lock();
        try
        {
            int total = 0;
            WordMapping mapping = new WordMapping();
            SensitiveWord word = null;
            while ((word = source.nextWord()) != null)
            {
                if (closed)
                {
                    return;
                }
                mapping.addWord(word);
                this.mapping = mapping;
                total++;
            }
            logger.info("Load SensitiveFilter with [" + total + "] words");
        }
        finally
        {
            loadLock.unlock();
        }
    }

    private void doLoad()
    {

        OracleSensitiveWordSource source = null;
        try
        {
            loadLock.lock();
            source = new OracleSensitiveWordSource(wordDb, domain);
            this.reload(source);
        }
        finally
        {
            loadLock.unlock();
            if (source != null) source.close();

        }
    }

    @Override
    public void close()
    {
        super.shutdown();
        closed = true;
    }

    static class WordMapping
    {

        private int minWordLen = Integer.MAX_VALUE;

        private int maxWordLen = 0;

        private Map<String, SensitiveWord> wordMap = new HashMap<String, SensitiveWord>();

        public void addWord(SensitiveWord word)
        {
            wordMap.put(word.getWord(), word);
            minWordLen = Math.min(word.getWord().length(), minWordLen);
            maxWordLen = Math.max(word.getWord().length(), maxWordLen);
        }

        public SensitiveWord hasWord(String word)
        {
            return wordMap.get(word);
        }

        public int getMaxWordLen()
        {
            return maxWordLen;
        }

        public boolean hasMapping()
        {
            return MapUtils.isNotEmpty(wordMap);
        }

        public int getMinWordLen()
        {
            return minWordLen;
        }

    }
}
