package com.aliyun.sdk.gateway.oss;

import com.aliyun.sdk.gateway.oss.auth.signer.OSSV1Signer;
import com.aliyun.sdk.gateway.oss.internal.OSSEndpointType;
import com.aliyun.sdk.gateway.oss.internal.interceptor.*;
import com.aliyun.sdk.gateway.oss.policy.OSSUserAgentPolicy;
import com.aliyun.sdk.gateway.oss.policy.retry.OSSRetryPolicy;
import darabonba.core.TeaClientBuilder;
import darabonba.core.client.ClientConfiguration;
import darabonba.core.client.ClientOption;
import darabonba.core.client.IClientBuilder;
import darabonba.core.interceptor.InterceptorChain;

import java.net.URI;
import java.net.URL;
import java.util.Optional;

public abstract class BaseClientBuilder<BuilderT extends IClientBuilder<BuilderT, ClientT>, ClientT> extends TeaClientBuilder<BuilderT, ClientT> {
    private String default_proto = "https";
    private String default_region = "cn-hangzhou";
    private Integer http_port = -1;

    @Override
    protected String serviceName() {
        return "OSS";
    }

    public BuilderT serviceConfiguration(Configuration serviceConfiguration) {
        clientConfiguration.setOption(ClientOption.SERVICE_CONFIGURATION, serviceConfiguration);
        return (BuilderT) this;
    }

    @Override
    protected ClientConfiguration mergeServiceDefaults(ClientConfiguration configuration) {
        return configuration.merge(ClientConfiguration.create()
                .setOption(ClientOption.SERVICE_CONFIGURATION, Configuration.create())
                .setOption(ClientOption.RETRY_POLICY, OSSRetryPolicy.defaultRetryPolicy())
                .setOption(ClientOption.CLOCK_SKEW_DIFF, 0L)
                .setOption(ClientOption.USER_AGENT_SERVICE_SUFFIX, OSSUserAgentPolicy.getDefaultUserAgentSuffix())
        );
    }

    @Override
    protected ClientConfiguration finalizeServiceConfiguration(ClientConfiguration configuration) {
        Configuration config = (Configuration) configuration.option(ClientOption.SERVICE_CONFIGURATION);

        //interceptor
        InterceptorChain chain = InterceptorChain.create();
        //request stage
        chain.addRequestInterceptor(new ProcPathRegexInterceptor());
        SelectObjectInterceptor selectObjectInterceptor = new SelectObjectInterceptor();
        chain.addRequestInterceptor(selectObjectInterceptor);
        chain.addRequestInterceptor(new TransformRequestBodyInterceptor());
        chain.addRequestInterceptor(new AddCommonHeadersInterceptor());
        ChecksumValidationInterceptor checksumInterceptor = new ChecksumValidationInterceptor();
        chain.addRequestInterceptor(checksumInterceptor);
        UrlDecodeEncodeInterceptor urlInterceptor = new UrlDecodeEncodeInterceptor();
        chain.addRequestInterceptor(urlInterceptor);

        //http request stage
        chain.addHttpRequestInterceptor(new MakeMutableHttpRequestInterceptor());
        chain.addHttpRequestInterceptor(new AddMetricInterceptor());
        chain.addHttpRequestInterceptor(new SigningInterceptor());
        chain.addHttpRequestInterceptor(new GenerateUrlInterceptor());
        //response stage
        chain.addResponseInterceptor(new MakeMutableResponseInterceptor());
        chain.addResponseInterceptor(new ProcResponseBodyInterceptor());
        chain.addResponseInterceptor(new AdjustClockSkew());
        chain.addResponseInterceptor(urlInterceptor);
        chain.addResponseInterceptor(checksumInterceptor);
        chain.addResponseInterceptor(selectObjectInterceptor);
        //output stage
        chain.addOutputInterceptor(new FinalizedOutputInterceptor());

        configuration.setOption(ClientOption.INTERCEPTOR_CHAIN, chain);

        //signer
        //config.signatureVersion() == SignVersion.V1
        //Signer signer;
        configuration.setOption(ClientOption.SIGNER, new OSSV1Signer());

        //region
        configuration.setOption(ClientOption.REGION, resolveRegion(configuration));

        //http port
        configuration.setOption(ClientOption.HTTP_PORT, resolveHttpPort(configuration));

        //http protocol
        configuration.setOption(ClientOption.HTTP_PROTOCOL, resolveHttpProtocol(configuration));

        //endpoint uri
        configuration.setOption(ClientOption.ENDPOINT_URI, resolveEndpointURI(configuration));

        return configuration;
    }

    private String resolveRegion(ClientConfiguration config) {
        return Optional.ofNullable(config.option(ClientOption.REGION)).orElse(default_region);
    }

    private Integer resolveHttpPort(ClientConfiguration config) {
        return Optional.ofNullable(config.option(ClientOption.HTTP_PORT)).orElse(http_port);
    }

    private String resolveHttpProtocol(ClientConfiguration config) {
        return Optional.ofNullable(config.option(ClientOption.HTTP_PROTOCOL)).orElse(default_proto);
    }

    private URI resolveEndpointURI(ClientConfiguration config) {
        return Optional.ofNullable(endpointURIFromEndpoint(config)).orElse(endpointURIFromRegion(config));
    }

    private URI toURI(String value) {
        URI uri = null;
        try {
            uri = new URI(value);
        } catch (Exception ignored) {
        }
        return uri;
    }

    private URI endpointURIFromEndpoint(ClientConfiguration config) {
        String value = config.option(ClientOption.ENDPOINT);
        String defaultProto = config.option(ClientOption.HTTP_PROTOCOL);
        if (value != null && !value.contains("://")) {
            value = defaultProto + "://" + value;
        }
        return toURI(value);
    }

    private URI endpointURIFromRegion(ClientConfiguration config) {
        final String region = config.option(ClientOption.REGION);
        final String type = Optional.ofNullable(config.option(ClientOption.ENDPOINT_TYPE)).orElse(OSSEndpointType.PUBLIC);
        final String proto = config.option(ClientOption.HTTP_PROTOCOL);
        String endpoint;
        switch (type.toLowerCase()) {
            case OSSEndpointType.INTERNAL:
            case OSSEndpointType.INTRANET:
                endpoint = "oss-" + region + "-internal.aliyuncs.com";
                break;
            case OSSEndpointType.ACCELERATE:
                endpoint = "oss-accelerate.aliyuncs.com";
                break;
            case OSSEndpointType.ACCELERATE_OVERSEAS:
                endpoint = "oss-accelerate-overseas.aliyuncs.com";
                break;
            case OSSEndpointType.DUAL_STACK:
                endpoint = region + ".oss.aliyuncs.com";
                break;
            case OSSEndpointType.PUBLIC:
            case OSSEndpointType.EXTRANET:
            default:
                endpoint = "oss-" + region + ".aliyuncs.com";
                break;
        }
        return toURI(proto + "://" + endpoint);
    }
}