package com.aliyun.sdk.gateway.oss.internal.interceptor;

import com.aliyun.core.http.Header;
import com.aliyun.core.utils.AttributeMap;
import com.aliyun.sdk.gateway.oss.util.Mimetypes;
import darabonba.core.TeaRequest;
import darabonba.core.TeaRequestBody;
import darabonba.core.client.ClientOption;
import darabonba.core.interceptor.InterceptorContext;
import darabonba.core.interceptor.RequestInterceptor;

import java.util.Arrays;
import java.util.List;

public class AddCommonHeadersInterceptor implements RequestInterceptor {
    public static final List<String> DETECTED_BY_OBJECT_NAME_ACTIONS = Arrays.asList(
            "PutObject",
            "AppendObject",
            "CopyObject",
            "InitiateMultipartUpload"
    );

    @Override
    public TeaRequest modifyRequest(InterceptorContext context, AttributeMap attributes) {
        TeaRequest request = context.teaRequest();
        addContentMD5IfNeeded(request, attributes);
        addContentTypeIfNeeded(request, attributes);
        addUserAgentIfNeeded(request, context);
        return request;
    }

    //Content-MD5
    private static void addContentMD5IfNeeded(TeaRequest request, AttributeMap attributes) {
        Header header = request.headers().get("Content-MD5");
        if (header == null && attributes.containsKey(AttributeKey.OSS_XML_BODY_CONTENT_MD5)) {
            String MD5 = attributes.get(AttributeKey.OSS_XML_BODY_CONTENT_MD5);
            request.headers().set("Content-MD5", MD5);
        }
    }

    //Content-Type
    private static void addContentTypeIfNeeded(TeaRequest request, AttributeMap attributes) {
        final String MIMETYPE_DEFAULT = "application/octet-stream";
        Header header = request.headers().get("Content-Type");
        if (header == null) {
            String contentType;
            TeaRequestBody requestBody = attributes.get(AttributeKey.REQUEST_BODY);
            if (requestBody.contentType().isPresent()) {
                contentType = requestBody.contentType().get();
            } else if (!DETECTED_BY_OBJECT_NAME_ACTIONS.contains(request.action())) {
                contentType = MIMETYPE_DEFAULT;
            } else {
                contentType = Mimetypes.getInstance().getMimetype(request.pathname());
            }
            request.headers().set("Content-Type", contentType);
        }
    }

    //User-Agent
    private static void addUserAgentIfNeeded(TeaRequest request, InterceptorContext context) {
        Header header = request.headers().get("User-Agent");
        if (header == null) {
            String ua = context.configuration().clientConfiguration().option(ClientOption.USER_AGENT);
            request.headers().set("User-Agent", ua);
        }
    }
}
