package com.aliyun.sdk.gateway.oss.internal.interceptor;

import com.aliyun.core.utils.AttributeMap;
import com.aliyun.sdk.gateway.oss.Configuration;
import com.aliyun.sdk.gateway.oss.exception.InconsistentException;
import com.aliyun.sdk.gateway.oss.internal.CRC64;
import com.aliyun.sdk.gateway.oss.internal.OSSHeaders;
import com.aliyun.sdk.gateway.oss.internal.async.CheckedAsyncResponseHandler;
import darabonba.core.TeaRequest;
import darabonba.core.TeaResponse;
import darabonba.core.async.AsyncResponseHandler;
import darabonba.core.interceptor.InterceptorContext;
import darabonba.core.interceptor.RequestInterceptor;
import darabonba.core.interceptor.ResponseInterceptor;

import java.math.BigInteger;
import java.util.Arrays;
import java.util.List;
import java.util.zip.CheckedInputStream;

public class ChecksumValidationInterceptor implements RequestInterceptor, ResponseInterceptor {
    private static final List<String> ALLOW_UPLOAD_ACTIONS = Arrays.asList(
            "PutObject",
            "UploadPart"
    );
    private static final List<String> ALLOW_DOWNLOAD_ACTIONS = Arrays.asList(
            "GetObject"
    );

    @Override
    public TeaRequest modifyRequest(InterceptorContext context, AttributeMap attributes) {
        TeaRequest request = context.teaRequest();
        final Configuration config = (Configuration) context.configuration().serviceConfiguration();
        if (config.crcEnabled()) {
            if (ALLOW_UPLOAD_ACTIONS.contains(request.action())) {
                attributes.put(AttributeKey.ENABLE_CHECKSUM_CRC64, Boolean.TRUE);
            } else if (ALLOW_DOWNLOAD_ACTIONS.contains(request.action())) {
                //GetObject needs to add crc handler into AsyncResponseHandler
                attributes.put(AttributeKey.ENABLE_CHECKSUM_CRC64, Boolean.TRUE);
                if (context.teaResponseHandler() instanceof AsyncResponseHandler) {
                    attributes.put(AttributeKey.OSS_ASYNC_RESPONSE_HANDLER,
                            new CheckedAsyncResponseHandler(context.teaResponseHandler(), new CRC64()));
                }
            }
        }
        return request;
    }

    @Override
    public TeaResponse modifyResponse(InterceptorContext context, AttributeMap attributes) {
        TeaResponse response = context.teaResponse();
        TeaRequest request = context.teaRequest();
        if (!response.success() || response.exception() != null) {
            return response;
        }

        boolean enableCRC64 = Boolean.TRUE.equals(attributes.get(AttributeKey.ENABLE_CHECKSUM_CRC64));
        if (enableCRC64) {
            Long clientCRC = null;
            String strServerCRC = response.httpResponse().getHeaders().getValue(OSSHeaders.HASH_CRC64_ECMA);
            if (ALLOW_UPLOAD_ACTIONS.contains(request.action())) {
                if (context.httpRequest().getStreamBody() instanceof CheckedInputStream) {
                    CheckedInputStream is = (CheckedInputStream) context.httpRequest().getStreamBody();
                    clientCRC = is.getChecksum().getValue();
                }
            } else if (ALLOW_DOWNLOAD_ACTIONS.contains(request.action())) {
                if (attributes.get(AttributeKey.OSS_ASYNC_RESPONSE_HANDLER) instanceof CheckedAsyncResponseHandler) {
                    CheckedAsyncResponseHandler handler = (CheckedAsyncResponseHandler) attributes.get(AttributeKey.OSS_ASYNC_RESPONSE_HANDLER);
                    clientCRC = handler.getChecksum().getValue();
                }
            }

            if (clientCRC != null && strServerCRC != null && !bypassCRCCheck(request)) {
                BigInteger bi = new BigInteger(strServerCRC);
                Long serverCRC = bi.longValue();
                if (!clientCRC.equals(serverCRC)) {
                    String requestId = response.httpResponse().getHeaders().getValue(OSSHeaders.REQUEST_ID);
                    response.setException(new InconsistentException(clientCRC, serverCRC, requestId));
                }
            }
        }
        return response;
    }

    private boolean bypassCRCCheck(TeaRequest request) {
        String value = request.headers().getValue("Range");
        if (value != null) {
            return true;
        }
        return false;
    }
}
