package com.aliyun.sdk.gateway.oss.internal.interceptor;

import com.aliyun.auth.credentials.ICredential;
import com.aliyun.auth.credentials.provider.ICredentialProvider;
import com.aliyun.core.utils.AttributeMap;
import com.aliyun.core.utils.StringUtils;
import com.aliyun.core.utils.Validate;
import com.aliyun.sdk.gateway.oss.auth.signer.OSSSigner;
import com.aliyun.sdk.gateway.oss.auth.signer.OSSSignerParams;
import darabonba.core.TeaRequest;
import darabonba.core.client.ClientConfiguration;
import darabonba.core.client.ClientOption;
import darabonba.core.interceptor.InterceptorContext;
import darabonba.core.interceptor.RequestInterceptor;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

public class PreSigningInterceptor implements RequestInterceptor {

    @Override
    public TeaRequest modifyRequest(InterceptorContext context, AttributeMap attributes) {
        final ClientConfiguration configuration = context.configuration().clientConfiguration();
        TeaRequest request = context.teaRequest();

        //Process query in path regex.
        List<String> extraSubresource = Optional.ofNullable(attributes.get(AttributeKey.EXTRA_SUBRESOURCE))
                .orElse(new ArrayList<>());

        //sign request
        OSSSigner signer = (OSSSigner) configuration.option(ClientOption.SIGNER);
        ICredentialProvider provider = configuration.option(ClientOption.CREDENTIALS_PROVIDER);
        ICredential cred = provider.getCredentials();
        Validate.notNull(cred, "ICredential is null");
        if (isAnonymous(cred)) {
            return request;
        }

        //EXPIRATION
        Instant instant;
        if (attributes.containsKey(AttributeKey.OSS_SIGNATURE_EXPIRATION)) {
            instant = attributes.get(AttributeKey.OSS_SIGNATURE_EXPIRATION);
        } else {
            instant = Instant.now().plusMillis(attributes.get(AttributeKey.OSS_SIGNATURE_DURATION).toMillis());
            attributes.put(AttributeKey.OSS_SIGNATURE_EXPIRATION, instant);
        }

        //add x-oss-security-token
        if (!StringUtils.isEmpty(cred.securityToken())) {
            request.query().put("security-token", cred.securityToken());
        }

        OSSSignerParams params = OSSSignerParams.create();
        params.setSubResources(extraSubresource);
        params.setCredentials(cred);
        params.setSignatureExpiration(instant);

        return signer.presign(request, params);
    }
    private boolean isAnonymous(ICredential cred) {
        if (cred == null ||
                StringUtils.isEmpty(cred.accessKeyId()) ||
                StringUtils.isEmpty(cred.accessKeySecret())) {
            return true;
        }
        return false;
    }
}
