package com.aliyun.sdk.gateway.oss.internal.interceptor;

import com.aliyun.core.http.BodyType;
import com.aliyun.core.http.HttpMethod;
import com.aliyun.core.http.HttpResponseHandler;
import com.aliyun.core.utils.AttributeMap;
import com.aliyun.core.utils.BaseUtils;
import com.aliyun.core.utils.XmlUtil;
import com.aliyun.sdk.gateway.oss.exception.OSSClientException;
import com.aliyun.sdk.gateway.oss.exception.OSSErrorDetails;
import com.aliyun.sdk.gateway.oss.exception.OSSServerException;
import com.aliyun.sdk.gateway.oss.internal.OSSHeaders;
import com.aliyun.sdk.gateway.oss.internal.async.StoredSeparatelyHttpResponseHandler;
import darabonba.core.ResponseBytes;
import darabonba.core.TeaRequest;
import darabonba.core.TeaResponse;
import darabonba.core.async.AsyncResponseHandler;
import darabonba.core.interceptor.InterceptorContext;
import darabonba.core.interceptor.ResponseInterceptor;
import darabonba.core.utils.CommonUtil;
import org.dom4j.DocumentException;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.*;

import static darabonba.core.internal.AttributeKey.HTTP_RESPONSE_HANDLER;

public class ProcResponseBodyInterceptor implements ResponseInterceptor {
    private static final List<String> HEAD_OBJECT_ACTIONS = Arrays.asList(
            "HeadObject",
            "GetObjectMeta"
    );

    @Override
    public TeaResponse modifyResponse(InterceptorContext context, AttributeMap attributes) {
        TeaResponse response = context.teaResponse();
        if (response.success()) {
            response = procSuccessResponse(context, attributes);
        } else {
            if (hasNotResponseBody(context)) {
                response = procFailResponseWithoutBody(context, attributes);
            } else {
                response = procFailResponse(context, attributes);
            }
        }
        return response;
    }

    private Object toDeserializedBodyForNoneType(InterceptorContext context, AttributeMap attributes) {
        TeaResponse response = context.teaResponse();
        Class<?> type = null;
        try {
            type = context.output().getClass().getDeclaredField("body").getType();
        } catch (Throwable e) {
        }
        if (type == String.class) {
            return response.httpResponse().getBodyAsString();
        } else if (type == InputStream.class) {
            return new ByteArrayInputStream(response.httpResponse().getBodyAsByteArray());
        }
        return response.httpResponse().getBodyAsByteArray();
    }

    private TeaResponse procSuccessResponse(InterceptorContext context, AttributeMap attributes) {
        TeaRequest request = context.teaRequest();
        TeaResponse response = context.teaResponse();
        if (skipDeserializeBody(context)) {
            //TODO don't parse body
            return response;
        }

        Exception exception = null;
        String bodyStr = "";
        try {
            switch (request.bodyType()) {
                case BodyType.XML:
                    bodyStr = response.httpResponse().getBodyAsString();
                    response.setDeserializedBody(deserializeXMLBody(bodyStr));
                    break;
                case BodyType.BIN:
                    response.setDeserializedBody(new ByteArrayInputStream(response.httpResponse().getBodyAsByteArray()));
                    break;
                case BodyType.NONE:
                    response.setDeserializedBody(toDeserializedBodyForNoneType(context, attributes));
                    break;
                default:
                    bodyStr = response.httpResponse().getBodyAsString();
                    response.setDeserializedBody(bodyStr);
                    break;
            }
        } catch (Exception e) {
            exception = new OSSClientException("Parsing response body fail, the partial text is " + partialString(bodyStr), e);
        }
        response.setException(exception);
        return response;
    }

    private TeaResponse procFailResponse(InterceptorContext context, AttributeMap attributes) {
        TeaResponse response = context.teaResponse();
        Exception exception = null;
        String bodyStr = "";
        String requestId = null;
        try {
            requestId = response.httpResponse().getHeaders().getValue(OSSHeaders.REQUEST_ID);
            bodyStr = readResponseErrorBody(response, attributes);
            Map<String, Object> body = deserializeXMLBody(bodyStr);
            OSSErrorDetails errorDetail;
            if (body.containsKey("Error")) {
                errorDetail = new OSSErrorDetails((HashMap) body.get("Error"), bodyStr);
            } else {
                errorDetail = buildFakeErrorDetails("InvalidResponseFormat", requestId, bodyStr);
            }
            exception = new OSSServerException(response.httpResponse().getStatusCode(), errorDetail);
        } catch (Exception e) {
            exception = new OSSServerException(response.httpResponse().getStatusCode(), buildFakeErrorDetails("InvalidResponseFormat", requestId, bodyStr));
        }
        response.setException(exception);
        return response;
    }

    private TeaResponse procFailResponseWithoutBody(InterceptorContext context, AttributeMap attributes) {
        String action = context.teaRequest().action();
        TeaResponse response = context.teaResponse();
        String requestId = response.httpResponse().getHeaders().getValue(OSSHeaders.REQUEST_ID);
        int code = response.httpResponse().getStatusCode();
        String strCode;
        if (HEAD_OBJECT_ACTIONS.contains(action) && code == 404) {
            //change 404 to NoSuchKey
            strCode = "NoSuchKey";
        } else {
            strCode = "Server:" + code;
        }
        HashMap map = new HashMap();
        map.put("Code", strCode);
        map.put("Message", "");
        map.put("RequestId", Optional.ofNullable(requestId).orElse(""));

        Exception exception = new OSSServerException(response.httpResponse().getStatusCode(),
                new OSSErrorDetails(map, ""));

        response.setException(exception);
        return response;
    }

    private boolean hasNotResponseBody(InterceptorContext context) {
        return (context.teaRequest().method() == HttpMethod.HEAD);
    }

    private String readResponseErrorBody(TeaResponse response, AttributeMap attributes) {
        if (attributes.containsKey(darabonba.core.internal.AttributeKey.HTTP_RESPONSE_HANDLER)) {
            HttpResponseHandler handler = attributes.get(darabonba.core.internal.AttributeKey.HTTP_RESPONSE_HANDLER);
            return BaseUtils.bomAwareToString(((StoredSeparatelyHttpResponseHandler) handler).getErrorBodyByteArrayUnsafe(), null);
        }

        return response.httpResponse().getBodyAsString();
    }

    private boolean skipDeserializeBody(InterceptorContext context) {
        return (context.teaResponseHandler() != null);
    }

    private Map<String, Object> deserializeXMLBody(String data) throws DocumentException {
        Object body = XmlUtil.deserializeXml(data);
        return CommonUtil.assertAsMap(Optional.ofNullable(body).orElse(new HashMap<>()));
    }

    private String partialString(String value) {
        int len = value.length() > 128 ? 128 : value.length();
        return value.substring(0, len);
    }

    private OSSErrorDetails buildFakeErrorDetails(String code, String requestId, String body) {
        if (body == null) {
            body = "";
        }
        HashMap map = new HashMap();
        String msg = "The response body is not well formed, the partial text is " + partialString(body);
        map.put("Code", code);
        map.put("Message", msg);
        map.put("RequestId", requestId);
        return new OSSErrorDetails(map, body);
    }

}
