package com.aliyun.sdk.gateway.oss.internal.interceptor;

import com.aliyun.core.http.HttpHeaders;
import com.aliyun.core.http.HttpResponseHandler;
import com.aliyun.core.utils.AttributeMap;
import com.aliyun.core.utils.Base64Util;
import com.aliyun.sdk.gateway.oss.exception.OSSClientException;
import com.aliyun.sdk.gateway.oss.exception.OSSErrorDetails;
import com.aliyun.sdk.gateway.oss.exception.OSSServerException;
import com.aliyun.sdk.gateway.oss.internal.OSSHeaders;
import darabonba.core.TeaRequest;
import darabonba.core.TeaResponse;
import darabonba.core.TeaResponseHandler;
import darabonba.core.async.AsyncResponseHandler;
import darabonba.core.interceptor.InterceptorContext;
import darabonba.core.interceptor.RequestInterceptor;
import darabonba.core.interceptor.ResponseInterceptor;
import org.reactivestreams.Processor;
import org.reactivestreams.Publisher;
import org.reactivestreams.Subscriber;
import org.reactivestreams.Subscription;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;
import java.util.zip.CRC32;

public class SelectObjectInterceptor implements RequestInterceptor, ResponseInterceptor {
    private static final List<String> REQUEST_ALLOW_ACTIONS = Arrays.asList(
            "SelectObject",
            "CreateSelectObjectMeta"
    );
    private static final AttributeKey<Boolean> SELECT_OBJECT_MEET_JSON = new AttributeKey<>(Boolean.class);
    private static final AttributeKey<Boolean> SELECT_OBJECT_HAS_ACTION = new AttributeKey<>(Boolean.class);
    public static final AttributeKey<HttpResponseHandler> SELECT_OBJECT_HTTP_RESPONSE_HANDLER = new AttributeKey<>(HttpResponseHandler.class);

    private static Object base64EncodeInputSerialization(HashMap root, AttributeMap attributes) {
        HashMap node = (HashMap) root.get("InputSerialization");
        if (node != null) {
            HashMap csv = (HashMap) node.get("CSV");
            HashMap json = (HashMap) node.get("JSON");
            if (csv != null) {
                base64EncodeIfPresent(csv, "RecordDelimiter");
                base64EncodeIfPresent(csv, "FieldDelimiter");
                base64EncodeIfPresent(csv, "QuoteCharacter");
                base64EncodeIfPresent(csv, "CommentCharacter");
                node.replace("CSV", csv);
            }

            if (json != null) {
                attributes.put(SELECT_OBJECT_MEET_JSON, Boolean.TRUE);
            }
        }
        return node;
    }

    private static Object base64EncodeOutputSerialization(HashMap root) {
        HashMap node = (HashMap) root.get("OutputSerialization");
        if (node != null) {
            HashMap csv = (HashMap) node.get("CSV");
            HashMap json = (HashMap) node.get("JSON");
            if (csv != null) {
                base64EncodeIfPresent(csv, "RecordDelimiter");
                base64EncodeIfPresent(csv, "FieldDelimiter");
                node.replace("CSV", csv);
            }
            if (json != null) {
                base64EncodeIfPresent(json, "RecordDelimiter");
                node.replace("JSON", json);
            }
        }
        return node;
    }

    private static void base64EncodeIfPresent(HashMap node, String key) {
        if (node.containsKey(key) && node.get(key) != null) {
            node.replace(key, Base64Util.encodeToString(((String) node.get(key)).getBytes()));
        }
    }


    @Override
    public TeaRequest modifyRequest(InterceptorContext context, AttributeMap attributes) {
        TeaRequest request = context.teaRequest();
        String action = request.action();
        if (REQUEST_ALLOW_ACTIONS.contains(action)) {
            // modify body
            HashMap body = (HashMap) request.body();
            if (body.containsKey("SelectRequest")) {
                HashMap root = (HashMap) body.get("SelectRequest");
                base64EncodeIfPresent(root, "Expression");
                root.replace("InputSerialization", base64EncodeInputSerialization(root, attributes));
                root.replace("OutputSerialization", base64EncodeOutputSerialization(root));
                body.replace("SelectRequest", root);

            } else if (body.containsKey("body")) {
                HashMap root = (HashMap) body.get("body");
                root.replace("InputSerialization", base64EncodeInputSerialization(root, attributes));
                String rootKey = attributes.containsKey(SELECT_OBJECT_MEET_JSON) ? "JsonMetaRequest" : "CsvMetaRequest";
                body.remove("body");
                body.put(rootKey, root);
            }
            request.setBody(body);

            //add x-oss-process parameter
            String op = attributes.containsKey(SELECT_OBJECT_MEET_JSON) ? "json/" : "csv/";
            op = op + (body.containsKey("SelectRequest") ? "select" : "meta");
            request.query().put(OSSHeaders.PROCESS, op);

            //
            attributes.put(SELECT_OBJECT_HAS_ACTION, Boolean.TRUE);

            // Add http handler to decode frame
            if (context.teaResponseHandler() instanceof AsyncResponseHandler) {
                HttpResponseHandler handler = new DecodeFrameHttpResponseHandler((AsyncResponseHandler<?, ?>) context.teaResponseHandler());
                attributes.put(AttributeKey.OSS_HTTP_RESPONSE_HANDLER, handler);
                attributes.put(SELECT_OBJECT_HTTP_RESPONSE_HANDLER, handler);
            }
        }
        return request;
    }

    private boolean shouldHandleResponse(TeaResponse response, AttributeMap attributes) {
        if (!response.success() || response.exception() != null) {
            return false;
        }
        return attributes.containsKey(SELECT_OBJECT_HAS_ACTION);
    }

    private static boolean hasSelectOutputRaw(HttpHeaders httpHeaders) {
        if (httpHeaders != null) {
            String value = httpHeaders.getValue(OSSHeaders.SELECT_OUTPUT_RAW);
            if ("true".equals(value)) {
                return true;
            }
        }
        return false;
    }

    private OSSClientException buildClientException(String msg) {
        return new OSSClientException(msg, null);
    }

    private OSSErrorDetails buildErrorDetails(String error, String requestId, int httpStatusCode) {
        HashMap map = new HashMap();
        int index = error.indexOf(".");
        if (index != -1) {
            map.put("Code", error.substring(0, index));
            map.put("Message", error.substring(index + 1));
        } else {
            map.put("Code", "");
            map.put("Message", error);
        }
        map.put("RequestId", requestId);
        map.put("HttpStatusCode", httpStatusCode + "");
        return new OSSErrorDetails(map, "");
    }

    private OSSServerException buildServerException(TeaResponse response, EndFrame frame) {
        String requestId = Optional.ofNullable(response.httpResponse().getHeaders().getValue(OSSHeaders.REQUEST_ID)).orElse("");
        return new OSSServerException(response.httpResponse().getStatusCode(), buildErrorDetails(frame.getErrorMessage(), requestId, frame.httpStatusCode));
    }

    private Exception changeDecodeFrameStatusToExceptionIfNeed(TeaResponse response, DecodeFrameStatus status) {
        Exception exception = null;
        if (status != null) {
            //check parsing status first
            if (status.getParsingError() != null) {
                //build client error
                exception = buildClientException(status.getParsingError());
            } else {
                // check endFrame http status code
                EndFrame endFrame = status.getEndFrame();
                if (endFrame != null) {
                    int httpStatusCode = endFrame.httpStatusCode;
                    if (httpStatusCode / 100 != 2) {
                        exception = buildServerException(response, endFrame);
                    }
                }
            }
        }
        return exception;
    }

    private TeaResponse modifySelectObjectResponse(InterceptorContext context, AttributeMap attributes) {
        TeaResponse response = context.teaResponse();
        Exception exception = null;
        if (context.teaResponseHandler() instanceof AsyncResponseHandler) {
            DecodeFrameHttpResponseHandler handler = (DecodeFrameHttpResponseHandler) attributes.get(SELECT_OBJECT_HTTP_RESPONSE_HANDLER);
            DecodeFrameStatus status = handler.getDecodeFrameStatus();
            exception = changeDecodeFrameStatusToExceptionIfNeed(response, status);
        } else if (response.deserializedBody() != null && !hasSelectOutputRaw(response.httpResponse().getHeaders())) {
            Object body = response.deserializedBody();
            if (body instanceof InputStream) {
                response.setDeserializedBody(new SelectInputStream((InputStream) body));
            }
        }
        response.setException(exception);
        return response;
    }

    private TeaResponse modifyCreateSelectObjectMetaResponse(TeaResponse response) {
        Exception exception = null;
        if (response.deserializedBody() instanceof byte[]) {
            CreateSelectMetaParser parser = new CreateSelectMetaParser();
            DecodeFrameStatus status = parser.parse((byte[]) response.deserializedBody());
            response.setDeserializedBody(parser.toEndFrameMap());
            exception = changeDecodeFrameStatusToExceptionIfNeed(response, status);
        }
        response.setException(exception);
        return response;
    }

    @Override
    public TeaResponse modifyResponse(InterceptorContext context, AttributeMap attributes) {
        TeaResponse response = context.teaResponse();
        if (shouldHandleResponse(response, attributes)) {
            switch (context.teaRequest().action()) {
                case "SelectObject":
                    response = modifySelectObjectResponse(context, attributes);
                    break;
                case "CreateSelectObjectMeta":
                    response = modifyCreateSelectObjectMetaResponse(response);
                    break;
                default:
                    break;
            }
        }
        return response;
    }

    private class SelectInputStream extends FilterInputStream {
        /**
         * Format of data frame
         * |--frame type(4 bytes)--|--payload length(4 bytes)--|--header checksum(4 bytes)--|
         * |--scanned data bytes(8 bytes)--|--payload--|--payload checksum(4 bytes)--|
         */
        private static final int DATA_FRAME_MAGIC = 8388609;

        /**
         * Format of continuous frame
         * |--frame type(4 bytes)--|--payload length(4 bytes)--|--header checksum(4 bytes)--|
         * |--scanned data bytes(8 bytes)--|--payload checksum(4 bytes)--|
         */
        private static final int CONTINUOUS_FRAME_MAGIC = 8388612;

        /**
         * Format of end frame
         * |--frame type(4 bytes)--|--payload length(4 bytes)--|--header checksum(4 bytes)--|
         * |--scanned data bytes(8 bytes)--|--total scan size(8 bytes)--|
         * |--status code(4 bytes)--|--error message--|--payload checksum(4 bytes)--|
         */
        private static final int END_FRAME_MAGIC = 8388613;
        private static final int SELECT_VERSION = 1;
        private static final long DEFAULT_NOTIFICATION_THRESHOLD = 50 * 1024 * 1024;//notify every scanned 50MB

        private long currentFrameOffset;
        private long currentFramePayloadLength;
        private byte[] currentFrameTypeBytes;
        private byte[] currentFramePayloadLengthBytes;
        private byte[] currentFrameHeaderChecksumBytes;
        private byte[] scannedDataBytes;
        private byte[] currentFramePayloadChecksumBytes;
        private boolean finished;
        private long nextNotificationScannedSize;
        private boolean payloadCrcEnabled;
        private CRC32 crc32;
        private String requestId;
        /**
         * payload checksum is the last 4 bytes in one frame, we use this flag to indicate whether we
         * need read the 4 bytes before we advance to next frame.
         */
        private boolean firstReadFrame;

        /**
         * Creates a <code>FilterInputStream</code>
         * by assigning the  argument <code>in</code>
         * to the field <code>this.in</code> so as
         * to remember it for later use.
         *
         * @param in the underlying input stream, or <code>null</code> if
         *           this instance is to be created without an underlying stream.
         */
        public SelectInputStream(InputStream in) {
            super(in);
            currentFrameOffset = 0;
            currentFramePayloadLength = 0;
            currentFrameTypeBytes = new byte[4];
            currentFramePayloadLengthBytes = new byte[4];
            currentFrameHeaderChecksumBytes = new byte[4];
            scannedDataBytes = new byte[8];
            currentFramePayloadChecksumBytes = new byte[4];
            finished = false;
            firstReadFrame = true;
            this.nextNotificationScannedSize = DEFAULT_NOTIFICATION_THRESHOLD;
            this.payloadCrcEnabled = true;
            if (this.payloadCrcEnabled) {
                this.crc32 = new CRC32();
                this.crc32.reset();
            }
        }

        private void internalRead(byte[] buf, int off, int len) throws IOException {
            int bytesRead = 0;
            while (bytesRead < len) {
                int bytes = in.read(buf, off + bytesRead, len - bytesRead);
                if (bytes < 0) {
                    throw new IOException("Invalid input stream end found, need another " + (len - bytesRead) + " bytes");
                }
                bytesRead += bytes;
            }
        }

        private void validateCheckSum(byte[] checksumBytes, CRC32 crc32) throws IOException {
            if (payloadCrcEnabled) {
                int currentChecksum = ByteBuffer.wrap(checksumBytes).getInt();
                if (currentChecksum != 0 && crc32.getValue() != ((long) currentChecksum & 0xffffffffL)) {
                    throw new IOException("Frame crc check failed, actual " + crc32.getValue() + ", expect: " + currentChecksum);
                }
                crc32.reset();
            }
        }

        private void readFrame() throws IOException {
            while (currentFrameOffset >= currentFramePayloadLength && !finished) {
                if (!firstReadFrame) {
                    internalRead(currentFramePayloadChecksumBytes, 0, 4);
                    validateCheckSum(currentFramePayloadChecksumBytes, crc32);
                }
                firstReadFrame = false;
                //advance to next frame
                internalRead(currentFrameTypeBytes, 0, 4);
                //first byte is version byte
                if (currentFrameTypeBytes[0] != SELECT_VERSION) {
                    throw new IOException("Invalid select version found " + currentFrameTypeBytes[0] + ", expect: " + SELECT_VERSION);
                }
                internalRead(currentFramePayloadLengthBytes, 0, 4);
                internalRead(currentFrameHeaderChecksumBytes, 0, 4);
                internalRead(scannedDataBytes, 0, 8);
                if (payloadCrcEnabled) {
                    crc32.update(scannedDataBytes, 0, scannedDataBytes.length);
                }

                currentFrameTypeBytes[0] = 0;
                int type = ByteBuffer.wrap(currentFrameTypeBytes).getInt();
                switch (type) {
                    case DATA_FRAME_MAGIC:
                        currentFramePayloadLength = ByteBuffer.wrap(currentFramePayloadLengthBytes).getInt() - 8;
                        currentFrameOffset = 0;
                        break;
                    case CONTINUOUS_FRAME_MAGIC:
                        //just break, continue
                        break;
                    case END_FRAME_MAGIC:
                        currentFramePayloadLength = ByteBuffer.wrap(currentFramePayloadLengthBytes).getInt() - 8;
                        byte[] totalScannedDataSizeBytes = new byte[8];
                        internalRead(totalScannedDataSizeBytes, 0, 8);
                        byte[] statusBytes = new byte[4];
                        internalRead(statusBytes, 0, 4);
                        if (payloadCrcEnabled) {
                            crc32.update(totalScannedDataSizeBytes, 0, totalScannedDataSizeBytes.length);
                            crc32.update(statusBytes, 0, statusBytes.length);
                        }
                        int status = ByteBuffer.wrap(statusBytes).getInt();
                        int errorMessageSize = (int) (currentFramePayloadLength - 12);
                        String error = "";
                        if (errorMessageSize > 0) {
                            byte[] errorMessageBytes = new byte[errorMessageSize];
                            internalRead(errorMessageBytes, 0, errorMessageSize);
                            error = new String(errorMessageBytes);
                            if (payloadCrcEnabled) {
                                crc32.update(errorMessageBytes, 0, errorMessageBytes.length);
                            }
                        }
                        finished = true;
                        currentFramePayloadLength = currentFrameOffset;
                        internalRead(currentFramePayloadChecksumBytes, 0, 4);

                        validateCheckSum(currentFramePayloadChecksumBytes, crc32);
                        if (status / 100 != 2) {
                            if (error.contains(".")) {
                                throw new IOException(error.substring(error.indexOf(".") + 1));
                            } else {
                                // forward compatbility consideration
                                throw new IOException(error);
                            }
                        }
                        break;
                    default:
                        throw new IOException("Unsupported frame type " + type + " found");
                }
                long scannedDataSize = ByteBuffer.wrap(scannedDataBytes).getLong();
                if (scannedDataSize >= nextNotificationScannedSize || finished) {
                    nextNotificationScannedSize += DEFAULT_NOTIFICATION_THRESHOLD;
                }
            }
        }

        @Override
        public int read() throws IOException {
            readFrame();
            int byteRead = in.read();
            if (byteRead >= 0) {
                currentFrameOffset++;
                if (payloadCrcEnabled) {
                    crc32.update(byteRead);
                }
            }
            return byteRead;
        }

        @Override
        public int read(byte b[]) throws IOException {
            return read(b, 0, b.length);
        }

        @Override
        public int read(byte[] buf, int off, int len) throws IOException {
            readFrame();
            int bytesToRead = (int) Math.min(len, currentFramePayloadLength - currentFrameOffset);
            if (bytesToRead != 0) {
                int bytes = in.read(buf, off, bytesToRead);
                if (bytes > 0) {
                    currentFrameOffset += bytes;
                    if (payloadCrcEnabled) {
                        crc32.update(buf, off, bytes);
                    }
                }
                return bytes;
            }
            return -1;
        }

        @Override
        public int available() throws IOException {
            throw new IOException("Select object input stream does not support available() operation");
        }
    }


    private class EndFrame {
        protected Long offset;
        protected Long totalScannedBytes;
        protected Integer httpStatusCode = 0;
        protected String errorMessage = "";

        public Long getOffset() {
            return offset;
        }

        public void setOffset(Long offset) {
            this.offset = offset;
        }

        public Long getTotalScannedBytes() {
            return totalScannedBytes;
        }

        public void setTotalScannedBytes(Long totalScannedBytes) {
            this.totalScannedBytes = totalScannedBytes;
        }

        public Integer getHttpStatusCode() {
            return httpStatusCode;
        }

        public void setHttpStatusCode(Integer httpStatusCode) {
            this.httpStatusCode = httpStatusCode;
        }

        public String getErrorMessage() {
            return errorMessage;
        }

        public void setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
        }
    }

    private class MetaEndFrame extends EndFrame {
        protected Integer splitsCount;
        protected Long rowsCount;
        protected Integer colsCount;

        public Integer getSplitsCount() {
            return splitsCount;
        }

        public void setSplitsCount(Integer splitsCount) {
            this.splitsCount = splitsCount;
        }

        public Long getRowsCount() {
            return rowsCount;
        }

        public void setRowsCount(Long rowsCount) {
            this.rowsCount = rowsCount;
        }

        public Integer getColsCount() {
            return colsCount;
        }

        public void setColsCount(Integer colsCount) {
            this.colsCount = colsCount;
        }
    }

    private class DecodeFrameStatus {
        private String parsingError;
        private EndFrame endFrame;

        public String getParsingError() {
            return parsingError;
        }

        public void setParsingError(String parsingError) {
            this.parsingError = parsingError;
        }

        public EndFrame getEndFrame() {
            return endFrame;
        }

        public void setEndFrame(EndFrame endFrame) {
            this.endFrame = endFrame;
        }
    }

    private class DecodeFrameParser {
        //Version | Frame - Type | Payload Length | Header Checksum | Payload | Payload Checksum
        //<1 byte> <--3 bytes-->   <-- 4 bytes --> <------4 bytes--> <variable><----4bytes------>
        //Payload
        //<offset | data>
        //<8 types><variable>
        protected static final int SELECT_VERSION = 1;
        protected static final int DATA_FRAME_MAGIC = 8388609;
        protected static final int CONTINUOUS_FRAME_MAGIC = 8388612;
        protected static final int END_FRAME_MAGIC = 8388613;
        protected static final int CSV_END_FRAME_MAGIC = 8388614;
        protected static final int JSON_END_FRAME_MAGIC = 8388615;

        protected static final int FRAME_HEADER_LEN = 12 + 8;  // header + offset

        protected byte[] headerBuf = new byte[FRAME_HEADER_LEN];
        protected byte[] tailBuf = new byte[4];
        protected byte[] endFramelBuf;

        protected int frameType;
        protected int headerLen;
        protected int tailLen;

        protected int payloadRemains;
        protected int payloadOff;
        protected int payloadLen;

        protected CRC32 crc32 = new CRC32();

        protected long payloadCRC;
        protected long calcPayloadCRC;

        protected String lastErrorMsg;

        DecodeFrameParser() {
        }

        protected void resetState() {
            headerLen = 0;
            tailLen = 0;
            payloadRemains = 0;
            lastErrorMsg = null;
            endFramelBuf = null;
        }

        protected int splitOneFrame(byte[] b, int off, int len) {
            int remians = len;
            int boff = off;

            // header
            if (headerLen < FRAME_HEADER_LEN) {
                int copy = Math.min(FRAME_HEADER_LEN - headerLen, remians);
                System.arraycopy(b, boff, headerBuf, headerLen, copy);
                headerLen += copy;
                boff += copy;
                remians -= copy;

                if (headerLen == FRAME_HEADER_LEN) {
                    int value = ByteBuffer.wrap(headerBuf, 4, 4).getInt();
                    payloadRemains = value - 8;
                    crc32.reset();
                    crc32.update(headerBuf, 12, 8);

                    //first byte is version byte
                    if (headerBuf[0] != SELECT_VERSION) {
                        lastErrorMsg = "Invalid select version found " + headerBuf[0] + ", expect: " + SELECT_VERSION;
                        return -1;
                    }
                }
            }

            //payload
            if (payloadRemains > 0) {
                int copy = Math.min(payloadRemains, remians);
                frameType = ByteBuffer.wrap(headerBuf, 0, 4).getInt();
                frameType &= 0x00FFFFFF;
                payloadOff = boff;
                payloadLen = copy;

                payloadRemains -= copy;
                boff += copy;
                remians -= copy;

                crc32.update(b, payloadOff, copy);
                return len - remians;
            }

            //tail
            if (tailLen < 4) {
                int copy = Math.min(4 - tailLen, remians);
                System.arraycopy(b, boff, tailBuf, tailLen, copy);

                tailLen += copy;
                remians -= copy;
                boff += copy;

                if (tailLen == 4) {
                    payloadCRC = ((long) ByteBuffer.wrap(tailBuf).getInt()) & 0xffffffffL;
                    calcPayloadCRC = crc32.getValue();
                }
            }
            return len - remians;
        }
    }

    private class DecodeFrameAsyncResponseHandler implements AsyncResponseHandler<String, DecodeFrameStatus> {
        protected volatile AsyncResponseHandler<?, ?> handler;
        protected volatile DecodeFrameProcessor processor;

        public DecodeFrameAsyncResponseHandler(TeaResponseHandler handler) {
            this.handler = (AsyncResponseHandler<?, ?>) handler;
        }

        @Override
        public void onStream(Publisher<ByteBuffer> publisher) {
            DecodeFrameProcessor proc = new DecodeFrameProcessor();
            this.processor = proc;
            this.handler.onStream(proc);
            publisher.subscribe(proc);
        }

        @Override
        public void onError(Throwable throwable) {
            if (this.handler != null) {
                this.handler.onError(throwable);
            }
        }

        @Override
        public DecodeFrameStatus transform(String response) {
            return this.processor.getDecodeFrameStatus();
        }

        class DecodeFrameProcessor extends DecodeFrameParser implements Processor<ByteBuffer, ByteBuffer> {
            protected volatile Subscriber<? super ByteBuffer> subscriber;
            private Subscription subscription;
            protected DecodeFrameStatus decodeFrameStatus;

            @Override
            protected void resetState() {
                super.resetState();
                decodeFrameStatus = null;
            }

            private boolean flushToSubscriber(ByteBuffer byteBuffer) {
                byte[] b = byteBuffer.array();
                int off = byteBuffer.arrayOffset() + byteBuffer.position();
                int remians = byteBuffer.remaining();
                int ret;

                while (remians > 0) {
                    frameType = 0;
                    payloadLen = 0;
                    payloadOff = 0;
                    if ((ret = splitOneFrame(b, off, remians)) < 0) {
                        return false;
                    }
                    switch (frameType) {
                        //DATA FRAME
                        case DATA_FRAME_MAGIC:
                            this.subscriber.onNext(ByteBuffer.wrap(b, payloadOff, payloadLen));
                            break;
                        // Select object End Frame
                        case END_FRAME_MAGIC:
                            endFramelBuf = new byte[payloadLen + 8];
                            System.arraycopy(headerBuf, 12, endFramelBuf, 0, 8);
                            System.arraycopy(b, payloadOff, endFramelBuf, 8, payloadLen);
                            break;
                        case 0:
                            // get payload checksum
                            if (tailLen == 4) {
                                if (payloadCRC != 0 && payloadCRC != calcPayloadCRC) {
                                    lastErrorMsg = "Frame crc check failed, actual " + calcPayloadCRC + ", expect: " + payloadCRC;
                                    return false;
                                }
                                //reset to next frame
                                headerLen = 0;
                                tailLen = 0;
                                payloadRemains = 0;
                            }
                            break;
                        default:
                            break;
                    }
                    remians -= ret;
                    off += ret;
                }

                return true;
            }

            private EndFrame toEndFrame(byte[] data) {
                if (data == null) {
                    return null;
                }
                EndFrame endFrame = new EndFrame();
                endFrame.setOffset(ByteBuffer.wrap(data, 0, 8).getLong());
                endFrame.setTotalScannedBytes(ByteBuffer.wrap(data, 8, 8).getLong());
                endFrame.setHttpStatusCode(ByteBuffer.wrap(data, 16, 4).getInt());
                endFrame.setErrorMessage(new String(data, 20, data.length - 20));
                return endFrame;
            }

            @Override
            public void subscribe(final Subscriber<? super ByteBuffer> subscriber) {
                this.subscriber = subscriber;
            }

            @Override
            public void onSubscribe(Subscription subscription) {
                resetState();
                this.subscription = subscription;
                this.subscriber.onSubscribe(subscription);
            }

            @Override
            public void onNext(ByteBuffer byteBuffer) {
                if (!flushToSubscriber(byteBuffer)) {
                    //this.subscription.cancel();
                }
            }

            @Override
            public void onError(Throwable throwable) {
                this.subscriber.onError(throwable);
            }

            @Override
            public void onComplete() {
                this.subscriber.onComplete();
                this.decodeFrameStatus = new DecodeFrameStatus();
                if (lastErrorMsg != null) {
                    this.decodeFrameStatus.setParsingError(lastErrorMsg);
                }
                if (endFramelBuf != null) {
                    this.decodeFrameStatus.setEndFrame(toEndFrame(endFramelBuf));
                }
            }

            public DecodeFrameStatus getDecodeFrameStatus() {
                return this.decodeFrameStatus;
            }
        }
    }

    private class DecodeFrameHttpResponseHandler implements HttpResponseHandler {
        private AsyncResponseHandler<?, ?> handler;
        private DecodeFrameAsyncResponseHandler decodeFrameHandler;

        DecodeFrameHttpResponseHandler(AsyncResponseHandler<?, ?> handler) {
            this.handler = handler;
            this.decodeFrameHandler = null;
        }

        @Override
        public void onStream(Publisher<ByteBuffer> publisher, int httpStatusCode, HttpHeaders headers) {
            if (hasSelectOutputRaw(headers)) {
                this.handler.onStream(publisher);
            } else {
                this.decodeFrameHandler = new DecodeFrameAsyncResponseHandler(this.handler);
                this.decodeFrameHandler.onStream(publisher);
            }
        }

        @Override
        public void onError(Throwable throwable) {
            if (this.decodeFrameHandler != null) {
                this.decodeFrameHandler.onError(throwable);
            } else if (this.handler != null) {
                this.handler.onError(throwable);
            }
        }

        public DecodeFrameStatus getDecodeFrameStatus() {
            if (this.decodeFrameHandler != null) {
                return this.decodeFrameHandler.transform("");
            } else {
                return null;
            }
        }
    }

    private class CreateSelectMetaParser extends DecodeFrameParser {
        private int metaEndframeType = 0;
        private MetaEndFrame metaEndFrame;

        @Override
        protected void resetState() {
            super.resetState();
            metaEndFrame = null;
        }

        private boolean parseInternal(byte[] b) {
            int off = 0;
            int remians = b.length;
            int ret;

            while (remians > 0) {
                frameType = 0;
                payloadLen = 0;
                payloadOff = 0;
                if ((ret = splitOneFrame(b, off, remians)) < 0) {
                    return false;
                }
                switch (frameType) {
                    case CSV_END_FRAME_MAGIC:
                    case JSON_END_FRAME_MAGIC:
                        metaEndframeType = frameType;
                        endFramelBuf = new byte[payloadLen + 8];
                        System.arraycopy(headerBuf, 12, endFramelBuf, 0, 8);
                        System.arraycopy(b, payloadOff, endFramelBuf, 8, payloadLen);
                        break;
                    case 0:
                        // get payload checksum
                        if (tailLen == 4) {
                            if (payloadCRC != 0 && payloadCRC != calcPayloadCRC) {
                                lastErrorMsg = "Frame crc check failed, actual " + calcPayloadCRC + ", expect: " + payloadCRC;
                                return false;
                            }
                            //reset to next frame
                            headerLen = 0;
                            tailLen = 0;
                            payloadRemains = 0;
                        }
                        break;
                    default:
                        break;
                }
                remians -= ret;
                off += ret;
            }
            return true;
        }

        private MetaEndFrame toMetaEndFrame(byte[] data) {
            if (data == null) {
                return null;
            }
            MetaEndFrame endFrame = new MetaEndFrame();
            endFrame.setOffset(ByteBuffer.wrap(data, 0, 8).getLong());
            endFrame.setTotalScannedBytes(ByteBuffer.wrap(data, 8, 8).getLong());
            endFrame.setHttpStatusCode(ByteBuffer.wrap(data, 16, 4).getInt());
            endFrame.setSplitsCount(ByteBuffer.wrap(data, 20, 4).getInt());
            endFrame.setRowsCount(ByteBuffer.wrap(data, 24, 8).getLong());
            if (metaEndframeType == CSV_END_FRAME_MAGIC) {
                endFrame.setColsCount(ByteBuffer.wrap(data, 32, 4).getInt());
                endFrame.setErrorMessage(new String(data, 36, data.length - 36));
            } else {
                endFrame.setErrorMessage(new String(data, 32, data.length - 32));
            }
            return endFrame;
        }

        public DecodeFrameStatus parse(byte[] data) {
            resetState();
            parseInternal(data);
            DecodeFrameStatus decodeFrameStatus = new DecodeFrameStatus();
            if (lastErrorMsg != null) {
                decodeFrameStatus.setParsingError(lastErrorMsg);
            }
            if (endFramelBuf != null) {
                this.metaEndFrame = toMetaEndFrame(endFramelBuf);
                decodeFrameStatus.setEndFrame(this.metaEndFrame);
            }
            return decodeFrameStatus;
        }

        public HashMap toEndFrameMap() {
            HashMap result = new HashMap();
            if (this.metaEndFrame != null) {
                result.put("Offset", this.metaEndFrame.getOffset());
                result.put("TotalScannedBytes", this.metaEndFrame.getTotalScannedBytes());
                result.put("Status", this.metaEndFrame.httpStatusCode);
                result.put("SplitsCount", this.metaEndFrame.splitsCount);
                result.put("RowsCount", this.metaEndFrame.getRowsCount());
                result.put("ColsCount", this.metaEndFrame.getColsCount());
                result.put("ErrorMessage", this.metaEndFrame.getErrorMessage());
            }
            return result;
        }
    }
}
