package com.aliyun.sdk.gateway.oss.internal.interceptor;

import com.aliyun.auth.credentials.ICredential;
import com.aliyun.auth.credentials.provider.ICredentialProvider;
import com.aliyun.core.http.HttpRequest;
import com.aliyun.core.utils.AttributeMap;
import com.aliyun.core.utils.DateUtil;
import com.aliyun.core.utils.StringUtils;
import com.aliyun.core.utils.Validate;
import com.aliyun.sdk.gateway.oss.auth.signer.OSSSigner;
import com.aliyun.sdk.gateway.oss.auth.signer.OSSSignerParams;
import com.aliyun.sdk.gateway.oss.internal.OSSHeaders;
import darabonba.core.TeaRequest;
import darabonba.core.client.ClientConfiguration;
import darabonba.core.client.ClientOption;
import darabonba.core.interceptor.HttpRequestInterceptor;
import darabonba.core.interceptor.InterceptorContext;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Optional;

public class SigningInterceptor implements HttpRequestInterceptor {

    @Override
    public HttpRequest modifyHttpRequest(InterceptorContext context, AttributeMap attributes) {
        final ClientConfiguration configuration = context.configuration().clientConfiguration();
        TeaRequest request = context.teaRequest();
        HttpRequest httpRequest = context.httpRequest();

        //Add Date
        if (httpRequest.getHeaders().getValue("Date") == null) {
            Date now = new Date();
            now.setTime(now.getTime() + configuration.option(ClientOption.CLOCK_SKEW_DIFF));
            httpRequest.getHeaders().set("Date", DateUtil.formatRfc822Date(now));
        }

        //Process query in path regex.
        List<String> extraSubresource = Optional.ofNullable(attributes.get(AttributeKey.EXTRA_SUBRESOURCE))
                .orElse(new ArrayList<>());

        //sign request
        OSSSigner signer = (OSSSigner) configuration.option(ClientOption.SIGNER);
        ICredentialProvider provider = configuration.option(ClientOption.CREDENTIALS_PROVIDER);
        ICredential cred = provider.getCredentials();
        Validate.notNull(cred, "ICredential is null");
        if (isAnonymous(cred)) {
            return httpRequest;
        }

        OSSSignerParams params = OSSSignerParams.create();
        params.setSubResources(extraSubresource);
        params.setCredentials(cred);

        //add x-oss-security-token
        if (!StringUtils.isEmpty(cred.securityToken())) {
            httpRequest.getHeaders().set(OSSHeaders.SECURITY_TOKEN, cred.securityToken());
        }
        return signer.sign(request, httpRequest, params);
    }

    private boolean isAnonymous(ICredential cred) {
        if (cred == null ||
                StringUtils.isEmpty(cred.accessKeyId()) ||
                StringUtils.isEmpty(cred.accessKeySecret())) {
            return true;
        }
        return false;
    }
}
