package com.aliyun.sdk.gateway.oss.internal.interceptor;

import com.aliyun.core.http.BodyType;
import com.aliyun.core.utils.AttributeMap;
import com.aliyun.core.utils.BinaryUtils;
import com.aliyun.core.utils.XmlUtil;
import darabonba.core.TeaRequest;
import darabonba.core.TeaRequestBody;
import darabonba.core.interceptor.InterceptorContext;
import darabonba.core.interceptor.RequestInterceptor;
import darabonba.core.sync.RequestBody;
import darabonba.core.utils.CommonUtil;

import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.Optional;

import static com.aliyun.sdk.gateway.oss.internal.interceptor.AddCommonHeadersInterceptor.DETECTED_BY_OBJECT_NAME_ACTIONS;

public class TransformRequestBodyInterceptor implements RequestInterceptor {
    private final static RequestBody EmptyBody = RequestBody.empty();

    private static String convertXMLRootName(String name) {
        return name.substring(0, 1).toUpperCase() + name.substring(1);
    }

    @Override
    public TeaRequest modifyRequest(InterceptorContext context, AttributeMap attributes) {
        TeaRequest request = context.teaRequest();
        if (request.body() != null) {
            return fromBody(context, attributes);
        } else if (context.teaRequestBody() != null) {
            return fromRequestBody(context, attributes);
        } else if (request.stream() != null) {
            return fromStream(context, attributes);
        } else {
            //request.headers().set("Content-Type", toContentType(request.reqBodyType()));
            setContentTypeIfNeeded(request, toContentType(request.reqBodyType()));
            attributes.put(AttributeKey.REQUEST_BODY, EmptyBody);
            return request;
        }
    }

    private TeaRequest fromBody(InterceptorContext context, AttributeMap attributes) {
        TeaRequest request = context.teaRequest();
        String bodyStr = null;
        switch (request.reqBodyType()) {
            case BodyType.XML: {
                Map<String, Object> body = CommonUtil.assertAsMap(request.body());
                if (!body.isEmpty()) {
                    String root = body.keySet().iterator().next();
                    bodyStr = XmlUtil.serializeXml(CommonUtil.assertAsMap(body.get(root)), convertXMLRootName(root));
                }
                calcContentMd5IfNeed(bodyStr, attributes);
            }
            break;
            default: {
                Map<String, Object> body = CommonUtil.assertAsMap(request.body());
                bodyStr = (String) body.get("body");
            }
            break;
        }
        if (bodyStr == null) {
            bodyStr = "";
        }
        attributes.put(AttributeKey.REQUEST_BODY, RequestBody.fromString(bodyStr, StandardCharsets.UTF_8, toContentType(request.reqBodyType())));
        //request.headers().set("Content-Type", toContentType(request.reqBodyType()));
        setContentTypeIfNeeded(request, toContentType(request.reqBodyType()));
        return request;
    }

    private TeaRequest fromStream(InterceptorContext context, AttributeMap attributes) {
        TeaRequest request = context.teaRequest();
        TeaRequestBody requestBody = RequestBody.fromInputStream(request.stream());
        attributes.put(AttributeKey.REQUEST_BODY, Optional.ofNullable(requestBody).orElse(EmptyBody));
        return request;
    }

    private TeaRequest fromRequestBody(InterceptorContext context, AttributeMap attributes) {
        TeaRequest request = context.teaRequest();
        TeaRequestBody requestBody = context.teaRequestBody();
        if (requestBody.contentLength().isPresent()) {
            attributes.put(AttributeKey.REQUEST_BODY_LENGTH, requestBody.contentLength().get());
        }
        attributes.put(AttributeKey.REQUEST_BODY, requestBody);
        return request;
    }

    private String toContentType(String type) {
        switch (type) {
            case BodyType.XML:
                return "application/xml";
            case BodyType.JSON:
                return "application/json";
            default:
                return "application/octet-stream";
        }
    }

    private void calcContentMd5IfNeed(String body, AttributeMap attributes) {
        if (body == null) {
            return;
        }
        byte[] md5 = BinaryUtils.calculateMd5(body.getBytes(StandardCharsets.UTF_8));
        String md5Base64 = BinaryUtils.toBase64(md5);
        attributes.put(AttributeKey.OSS_XML_BODY_CONTENT_MD5, md5Base64);
    }

    private void setContentTypeIfNeeded(TeaRequest request, String contentType) {
        if (DETECTED_BY_OBJECT_NAME_ACTIONS.contains(request.action())) {
            return;
        }
        request.headers().set("Content-Type", contentType);
    }
}
