package com.aliyun.sdk.gateway.oss.internal.interceptor;

import com.aliyun.core.http.HttpHeader;
import com.aliyun.core.http.HttpResponse;
import com.aliyun.core.utils.AttributeMap;
import com.aliyun.sdk.gateway.oss.internal.HttpUtil;
import com.aliyun.sdk.gateway.oss.internal.OSSHeaders;
import darabonba.core.TeaRequest;
import darabonba.core.TeaResponse;
import darabonba.core.interceptor.InterceptorContext;
import darabonba.core.interceptor.RequestInterceptor;
import darabonba.core.interceptor.ResponseInterceptor;

import java.util.*;
import java.util.stream.Collectors;

public class UrlDecodeEncodeInterceptor implements RequestInterceptor, ResponseInterceptor {
    private static final List<String> REQUEST_ALLOW_ACTIONS = Arrays.asList(
            "PutSymlink"
    );

    @Override
    public TeaRequest modifyRequest(InterceptorContext context, AttributeMap attributes) {
        TeaRequest request = context.teaRequest();
        final  String action = context.teaRequest().action();
        if (shouldHandleRequest(request, action)) {
            switch (action) {
                case "PutSymlink":
                    request = modifyPutSymlinkRequest(request);
                    break;
                default:
                    break;
            };
        }
        return request;
    }

    private static String urlEncode(String v) {
        return HttpUtil.urlEncode(v, "utf-8");
    }

    private boolean shouldHandleRequest(TeaRequest request, String action) {
        return REQUEST_ALLOW_ACTIONS.contains(action);
    }

    private TeaRequest modifyPutSymlinkRequest(TeaRequest request) {
        HttpHeader header = request.headers().get(OSSHeaders.SYMLINK_TARGET);
        if (header != null) {
            request.headers().set(OSSHeaders.SYMLINK_TARGET, urlEncode(header.getValue()));
        }
        return request;
    }


/////////////////////////////////////////////////////////////////////////////////////////////////
    private static final List<String> RESPONSE_ALLOW_ACTIONS = Arrays.asList(
            "ListObjects",
            "ListObjectsV2",
            "ListObjectVersions",
            "DeleteMultipleObjects",
            "ListMultipartUploads",
            "ListParts",
            "GetSymlink",
            "InitiateMultipartUpload",
            "CompleteMultipartUpload"
    );

    private static Object urlDecode(String v) {
        return HttpUtil.urlDecode(v, "utf-8");
    }

    private static ArrayList<?> urlDecodeInList(HashMap root, String node, String key) {
        Object obj = root.get(node);
        ArrayList<Object> list = null;
        if (List.class.isAssignableFrom(obj.getClass())) {
            list = (ArrayList<Object>) obj;
        } else {
            list = new ArrayList<Object>();
            list.add(obj);
        }

        return  (ArrayList<Object>) list.stream().map(o -> {
            HashMap m = (HashMap) o;
            m.replace(key, urlDecode((String) m.get(key)));
            return o;
        }).collect(Collectors.toList());
    }

    @Override
    public TeaResponse modifyResponse(InterceptorContext context, AttributeMap attributes) {
        TeaResponse response = context.teaResponse();
        String action = context.teaRequest().action();
        if (shouldHandleResponse(response, action)) {
            switch (action) {
                case "ListObjects":
                    response = modifyListObjectsResponse(response);
                    break;
                case "ListObjectsV2":
                    response = modifyListObjectsV2Response(response);
                    break;
                case "ListObjectVersions":
                    response = modifyListObjectVersionsResponse(response);
                    break;
                case "DeleteMultipleObjects":
                    response = modifyDeleteMultipleObjectsResponse(response);
                    break;
                case "ListMultipartUploads":
                    response = modifyListMultipartUploadsResponse(response);
                    break;
                case "ListParts":
                    response = modifyListPartsResponse(response);
                    break;
                case "GetSymlink":
                    response = modifyGetSymlinkResponse(response);
                    break;
                case "InitiateMultipartUpload":
                    response = modifyInitiateMultipartUploadResponse(response);
                    break;
                case "CompleteMultipartUpload":
                    response = modifyCompleteMultipartUploadResponse(response);
                    break;
                default:
                    break;
            }
        }
        return response;
    }

    private boolean shouldHandleResponse(TeaResponse response, String action) {
        if (!response.success() || response.exception() != null) {
            return false;
        }
        return RESPONSE_ALLOW_ACTIONS.contains(action);
    }

    // Delimiter, Marker, Prefix, NextMarker, Key
    private TeaResponse modifyListObjectsResponse(TeaResponse response) {
        HashMap body = (HashMap) response.deserializedBody();
        HashMap root = (HashMap) body.get("ListBucketResult");
        if (root != null && ((String) root.getOrDefault("EncodingType", "")).equals("url")) {
            final List<String> Keys = Arrays.asList("Delimiter", "Marker", "Prefix", "NextMarker");
            for (String key : Keys) {
                if (root.containsKey(key)) {
                    root.replace(key, urlDecode((String) root.get(key)));
                }
            }

            //Contents
            if (root.containsKey("Contents")) {
                root.replace("Contents", urlDecodeInList(root, "Contents", "Key"));
            }

            //CommonPrefixes
            if (root.containsKey("CommonPrefixes")) {
                root.replace("CommonPrefixes", urlDecodeInList(root, "CommonPrefixes", "Prefix"));
            }

            body.replace("ListBucketResult", root);
            response.setDeserializedBody(body);
        }

        return response;
    }

    // Delimiter, StartAfter, Prefix, Key
    private TeaResponse modifyListObjectsV2Response(TeaResponse response) {
        HashMap body = (HashMap) response.deserializedBody();
        HashMap root = (HashMap) body.get("ListBucketResult");
        if (root != null && ((String) root.getOrDefault("EncodingType", "")).equals("url")) {
            final List<String> Keys = Arrays.asList("Delimiter", "StartAfter", "Prefix");
            for (String key : Keys) {
                if (root.containsKey(key)) {
                    root.replace(key, urlDecode((String) root.get(key)));
                }
            }

            //Contents
            if (root.containsKey("Contents")) {
                root.replace("Contents", urlDecodeInList(root, "Contents", "Key"));
            }

            //CommonPrefixes
            if (root.containsKey("CommonPrefixes")) {
                root.replace("CommonPrefixes", urlDecodeInList(root, "CommonPrefixes", "Prefix"));
            }

            body.replace("ListBucketResult", root);
            response.setDeserializedBody(body);
        }

        return response;
    }

    // Delimiter, KeyMarker, NextKeyMarker, Prefix, Key
    private TeaResponse modifyListObjectVersionsResponse(TeaResponse response) {
        HashMap body = (HashMap) response.deserializedBody();
        HashMap root = (HashMap) body.get("ListVersionsResult");
        if (root != null && ((String) root.getOrDefault("EncodingType", "")).equals("url")) {
            final List<String> Keys = Arrays.asList("Delimiter", "KeyMarker", "NextKeyMarker", "Prefix");
            for (String key : Keys) {
                if (root.containsKey(key)) {
                    root.replace(key, urlDecode((String) root.get(key)));
                }
            }

            //Version
            if (root.containsKey("Version")) {
                root.replace("Version", urlDecodeInList(root, "Version", "Key"));
            }

            //DeleteMarker
            if (root.containsKey("DeleteMarker")) {
                root.replace("DeleteMarker", urlDecodeInList(root, "DeleteMarker", "Key"));
            }

            //CommonPrefixes
            if (root.containsKey("CommonPrefixes")) {
                root.replace("CommonPrefixes", urlDecodeInList(root, "CommonPrefixes", "Prefix"));
            }

            body.replace("ListVersionsResult", root);
            response.setDeserializedBody(body);
        }

        return response;
    }

    // Key
    private TeaResponse modifyDeleteMultipleObjectsResponse(TeaResponse response) {
        HashMap body = (HashMap) response.deserializedBody();
        HashMap root = (HashMap) body.get("DeleteResult");
        if (root != null && ((String) root.getOrDefault("EncodingType", "")).equals("url")) {
            //Deleted
            if (root.containsKey("Deleted")) {
                root.replace("Deleted", urlDecodeInList(root, "Deleted", "Key"));
            }

            body.replace("DeleteResult", root);
            response.setDeserializedBody(body);
        }
        return response;
    }

    // Delimiter, Prefix, KeyMarker, NextKeyMarker, Key
    private TeaResponse modifyListMultipartUploadsResponse(TeaResponse response) {
        HashMap body = (HashMap) response.deserializedBody();
        HashMap root = (HashMap) body.get("ListMultipartUploadsResult");
        if (root != null && ((String) root.getOrDefault("EncodingType", "")).equals("url")) {
            final List<String> Keys = Arrays.asList("Delimiter", "KeyMarker", "NextKeyMarker", "Prefix");
            for (String key : Keys) {
                if (root.containsKey(key)) {
                    root.replace(key, urlDecode((String) root.get(key)));
                }
            }

            //Upload
            if (root.containsKey("Upload")) {
                root.replace("Upload", urlDecodeInList(root, "Upload", "Key"));
            }

            //CommonPrefixes
            if (root.containsKey("CommonPrefixes")) {
                root.replace("CommonPrefixes", urlDecodeInList(root, "CommonPrefixes", "Prefix"));
            }

            body.replace("ListMultipartUploadsResult", root);
            response.setDeserializedBody(body);
        }
        return response;
    }

    // Key
    private TeaResponse modifyListPartsResponse(TeaResponse response) {
        HashMap body = (HashMap) response.deserializedBody();
        HashMap root = (HashMap) body.get("ListPartsResult");
        if (root != null && ((String) root.getOrDefault("EncodingType", "")).equals("url")) {
            final List<String> Keys = Arrays.asList("Key");
            for (String key : Keys) {
                if (root.containsKey(key)) {
                    root.replace(key, urlDecode((String) root.get(key)));
                }
            }

            body.replace("ListPartsResult", root);
            response.setDeserializedBody(body);
        }
        return response;
    }

    //x-oss-symlink-target
    private TeaResponse modifyGetSymlinkResponse(TeaResponse response) {
        HttpResponse httpResponse = response.httpResponse();
        HttpHeader header = httpResponse.getHeaders().get(OSSHeaders.SYMLINK_TARGET);
        if (header != null) {
            httpResponse.getHeaders().set(OSSHeaders.SYMLINK_TARGET, (String)urlDecode(header.getValue()));
        }
        return response;
    }

    private TeaResponse modifyInitiateMultipartUploadResponse(TeaResponse response) {
        HashMap body = (HashMap) response.deserializedBody();
        HashMap root = (HashMap) body.get("InitiateMultipartUploadResult");
        if (root != null && ((String) root.getOrDefault("EncodingType", "")).equals("url")) {
            final List<String> Keys = Arrays.asList("Key");
            for (String key : Keys) {
                if (root.containsKey(key)) {
                    root.replace(key, urlDecode((String) root.get(key)));
                }
            }

            body.replace("InitiateMultipartUploadResult", root);
            response.setDeserializedBody(body);
        }
        return response;
    }

    private TeaResponse modifyCompleteMultipartUploadResponse(TeaResponse response) {
        HashMap body = (HashMap) response.deserializedBody();
        HashMap root = (HashMap) body.get("CompleteMultipartUploadResult");
        if (root != null && ((String) root.getOrDefault("EncodingType", "")).equals("url")) {
            final List<String> Keys = Arrays.asList("Key");
            for (String key : Keys) {
                if (root.containsKey(key)) {
                    root.replace(key, urlDecode((String) root.get(key)));
                }
            }

            body.replace("CompleteMultipartUploadResult", root);
            response.setDeserializedBody(body);
        }
        return response;
    }
}
