package com.aliyun.sdk.gateway.oss.models;

import com.aliyun.sdk.gateway.oss.internal.OSSConstants;
import darabonba.core.RequestModel;
import darabonba.core.exception.ValidateException;

import java.io.UnsupportedEncodingException;

public abstract class Request extends RequestModel {

    protected Request(Builder<?, ?> builder) {
        super(builder);
    }

    @Override
    public void validate() {
        super.validate();
        ensureBucketNameValidIfNotNull(getHostParameters().get("bucket"));
        ensureObjectKeyValidIfNotNull(getPathParameters().get("key"));
    }

    private static boolean validateBucketName(String bucketName) {
        if (bucketName == null) {
            return false;
        }
        return bucketName.matches(OSSConstants.BUCKET_NAMING_REGEX);
    }

    private static void ensureBucketNameValidIfNotNull(String bucketName) {
        if (bucketName == null) {
            return;
        }
        if (!validateBucketName(bucketName)) {
            throw new ValidateException("The bucket name " + bucketName + " is invalid.");
        }
    }

    private static boolean validateObjectKey(String key) {
        if (key == null || key.length() == 0) {
            return true;
        }

        byte[] bytes;
        try {
            bytes = key.getBytes(OSSConstants.DEFAULT_CHARSET_NAME);
        } catch (UnsupportedEncodingException e) {
            return true;
        }

        // Validate exclude xml unsupported chars
        char[] keyChars = key.toCharArray();
        char firstChar = keyChars[0];
        if (firstChar == '\\') {
            return true;
        }
        return (bytes.length <= 0 || bytes.length >= OSSConstants.OBJECT_NAME_MAX_LENGTH);
    }

    private static void ensureObjectKeyValidIfNotNull(String key) {
        if (key == null) {
            return;
        }
        if (validateObjectKey(key)) {
            throw new ValidateException("The object name " + key + " is invalid.");
        }
    }

    public abstract Builder toBuilder();

    protected abstract static class Builder<ProviderT extends Request, BuilderT extends Builder>
            extends RequestModel.BuilderImpl<ProviderT, BuilderT> {

        protected Builder() {
        }

        protected Builder(Request request) {
        }

        public BuilderT headerParam(String name, Object value) {
            putHeaderParameter(name, value);
            return (BuilderT) this;
        }

        public BuilderT queryParam(String name, Object value) {
            putQueryParameter(name, value);
            return (BuilderT) this;
        }
    }

}
