package com.aliyun.auth.credentials.provider;

import com.aliyun.auth.credentials.ICredential;
import com.aliyun.auth.credentials.exception.*;
import com.aliyun.auth.credentials.utils.*;

import java.time.Duration;
import java.time.Instant;
import java.util.Optional;

public abstract class HttpCredentialProvider implements ICredentialProvider {
    private final boolean asyncCredentialUpdateEnabled;
    private Optional<RefreshCachedSupplier<ICredential>> credentialsCache;

    protected HttpCredentialProvider(BuilderImpl<?, ?> builder) {
        this.asyncCredentialUpdateEnabled = builder.asyncCredentialUpdateEnabled;
    }

    protected void buildRefreshCache(){
        RefreshCachedSupplier.Builder<ICredential> cacheBuilder = RefreshCachedSupplier.builder(this::refreshCredentials);
        if (this.asyncCredentialUpdateEnabled) {
            cacheBuilder.prefetchStrategy(new NonBlocking());
        }
        this.credentialsCache = Optional.of(cacheBuilder.build());
    }

    public abstract RefreshResult<ICredential> refreshCredentials();

    public Instant getStaleTime(Instant expiration) {
        return expiration == null ? null
                : expiration.minus(Duration.ofMinutes(1));
    }

    public Instant getPrefetchTime(Instant expiration) {
        Instant oneHourFromNow = Instant.now().plus(Duration.ofHours(1));
        if (expiration == null) {
            return oneHourFromNow;
        }
        Instant prefetchTime = expiration.minus(Duration.ofMinutes(15));
        return oneHourFromNow.isAfter(prefetchTime) ? prefetchTime : oneHourFromNow;
    }

    @Override
    public ICredential getCredentials() {
        return credentialsCache.map(RefreshCachedSupplier::get).orElseThrow(() ->
                new CredentialException("Unable to get credentials"));
    }

    @Override
    public void close() {
        credentialsCache.ifPresent(RefreshCachedSupplier::close);
    }

    public interface Builder<ProviderT extends HttpCredentialProvider, BuilderT extends Builder> {
        BuilderT asyncCredentialUpdateEnabled(Boolean asyncCredentialUpdateEnabled);

        ProviderT build();
    }

    protected abstract static class BuilderImpl<ProviderT extends HttpCredentialProvider, BuilderT extends Builder>
            implements Builder<ProviderT, BuilderT> {
        private boolean asyncCredentialUpdateEnabled = false;

        protected BuilderImpl() {
        }

        @Override
        public BuilderT asyncCredentialUpdateEnabled(Boolean asyncCredentialUpdateEnabled) {
            this.asyncCredentialUpdateEnabled = asyncCredentialUpdateEnabled;
            return (BuilderT) this;
        }
    }
}
