package darabonba.core;

import com.aliyun.core.http.*;
import com.aliyun.core.logging.ClientLogger;
import com.aliyun.core.utils.AttributeMap;
import com.aliyun.core.utils.Context;
import com.aliyun.core.utils.SdkAutoCloseable;
import com.aliyun.core.utils.Validate;
import com.aliyun.httpcomponent.httpclient.ApacheAsyncHttpClientBuilder;
import darabonba.core.client.ClientConfiguration;
import darabonba.core.client.ClientExecutionParams;
import darabonba.core.client.ClientOption;
import darabonba.core.client.IAsyncHandler;
import darabonba.core.exception.ClientException;
import darabonba.core.exception.TeaException;
import darabonba.core.interceptor.InterceptorChain;
import darabonba.core.interceptor.InterceptorContext;
import darabonba.core.internal.AttributeKey;
import darabonba.core.policy.retry.RetryPolicy;
import darabonba.core.policy.retry.RetryPolicyContext;
import darabonba.core.utils.CommonUtil;
import darabonba.core.utils.ModelUtil;

import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

public class TeaAsyncHandler implements IAsyncHandler, SdkAutoCloseable {
    private final String RESPONSE_HANDLER_KEY = "RESPONSE_HANDLER";
    private final ClientLogger logger = new ClientLogger(TeaAsyncHandler.class);
    private final ClientConfiguration configuration;
    private HttpClient httpClient;

    public TeaAsyncHandler(ClientConfiguration config) {
        this.httpClient = resolveHttpClient(config);
        Validate.notNull(config.option(ClientOption.CREDENTIALS_PROVIDER), "Credentials must not be null.");
        this.configuration = config;
    }

    private HttpClient resolveHttpClient(ClientConfiguration config) {
        return Optional.ofNullable(config.option(ClientOption.ASYNC_HTTP_CLIENT))
                .orElseGet(() -> new ApacheAsyncHttpClientBuilder()
                        .connectionTimeout(config.option(ClientOption.CONNECT_TIMEOUT))
                        .responseTimeout(config.option(ClientOption.RESPONSE_TIMEOUT))
                        .build());
    }

    private HttpRequest composeHttpRequest(InterceptorContext context) {
        TeaRequest request = context.teaRequest();
        TeaConfiguration configuration = context.configuration();
        HttpRequest httpRequest;
        if (CommonUtil.isUnset(httpRequest = context.httpRequest())) {
            httpRequest = new HttpRequest(
                    Optional.ofNullable(configuration.method()).orElseGet(request::method),
                    ModelUtil.composeUrl(configuration.endpoint(),
                            request.query(),
                            configuration.protocol(),
                            request.pathname()));
            HttpHeaders httpHeaders = new HttpHeaders(request.headers());
            httpRequest.setHeaders(httpHeaders);
            if (request.body() instanceof byte[]) {
                httpRequest.setBody((byte[]) request.body());
            }
        }
        httpRequest.setConnectTimeout(configuration.connectTimeout())
                .setResponseTimeout(configuration.responseTimeout());
//                .setReadTimeout(configuration.readTimeout())
//                .setWriteTimeout(configuration.writeTimeout());
        return httpRequest;
    }

    private Context composeHttpRequestContext(InterceptorContext context, AttributeMap attributes) {
        Context ctx = Context.NONE;

        //HTTP_RESPONSE_HANDLER
        HttpResponseHandler httpResponseHandler = null;
        if (attributes.containsKey(AttributeKey.HTTP_RESPONSE_HANDLER)) {
            httpResponseHandler = attributes.get(AttributeKey.HTTP_RESPONSE_HANDLER);
        } else if (context.httpResponseHandler() != null) {
            httpResponseHandler = context.httpResponseHandler();
        }
        if (httpResponseHandler != null) {
            ctx = ctx.addData(RESPONSE_HANDLER_KEY, httpResponseHandler);
        }
        return ctx;
    }

    public void validateRequestModel(RequestModel model) {
        if (CommonUtil.isUnset(model)) {
            throw new TeaException("RequestModel is not allowed as null",
                    new RuntimeException("RequestModel is not allowed as null"));
        }
        model.validate();
    }

    public HttpClient httpClient() {
        return this.httpClient;
    }

    public ClientConfiguration configuration() {
        return this.configuration;
    }

    @Override
    public void close() {
        this.configuration.close();
        this.httpClient.close();
    }

    public interface Builder<ProviderT, BuilderT extends Builder> {

        BuilderT httpClient(HttpClient httpClient);

        BuilderT configuration(ClientConfiguration configuration);

        ProviderT build();
    }

    protected abstract static class BuilderImpl<ProviderT, BuilderT extends Builder>
            implements Builder<ProviderT, BuilderT> {

        private HttpClient httpClient;
        private ClientConfiguration configuration = ClientConfiguration.create();

        @Override
        public BuilderT httpClient(HttpClient httpClient) {
            this.httpClient = httpClient;
            return (BuilderT) this;
        }

        @Override
        public BuilderT configuration(ClientConfiguration configuration) {
            this.configuration = configuration;
            return (BuilderT) this;
        }
    }

    @Override
    public <InputT extends RequestModel, OutputT extends TeaModel> CompletableFuture<OutputT> execute(
            ClientExecutionParams<InputT, OutputT> executionParams) {
        final InterceptorChain interceptorChain = configuration.option(ClientOption.INTERCEPTOR_CHAIN);
        final TeaConfiguration config = new TeaConfiguration(configuration, executionParams.getRequest().requestConfiguration());
        HttpHeaders headers = new HttpHeaders();
        headers.putAll(executionParams.getRequest().headers());
        headers.putAll(config.httpHeaders());
        TeaRequest request = executionParams.getRequest().copy().setHeaders(headers);
        InterceptorContext context = InterceptorContext.builder()
                .teaRequest(request)
                .teaRequestBody(executionParams.getRequestBody())
                .configuration(config)
                .build();
        context.setOutput(executionParams.getOutput());
        context.setTeaResponseHandler(executionParams.getResponseHandler());
        context.setHttpResponseHandler(executionParams.getHttpResponseHandler());
        AttributeMap attributes = AttributeMap.empty();
        context = interceptorChain.modifyConfiguration(context, attributes);
        context = interceptorChain.modifyRequest(context, attributes);
        RetryableExecutor executor = new RetryableExecutor(context, attributes, httpClient);
        return executor.execute()
                .thenCompose((output) -> CompletableFuture.completedFuture((OutputT) output));
    }

    private class RetryableExecutor {
        private final ClientConfiguration configuration;
        private final HttpClient httpClient;
        private final RetryPolicy retryPolicy;

        private int attemptNumber;
        private Throwable lastException;
        private HttpResponse lastHttpResponse;
        private RetryPolicyContext retryContext;
        private AttributeMap attributes;
        private InterceptorContext interceptorContext;

        private RetryableExecutor(InterceptorContext interceptorContext, AttributeMap attributes, HttpClient httpClient) {
            this.configuration = interceptorContext.configuration().clientConfiguration();
            this.httpClient = httpClient;
            this.retryPolicy = configuration.option(ClientOption.RETRY_POLICY);
            this.attemptNumber = 0;
            this.lastException = null;
            this.lastHttpResponse = null;
            this.attributes = attributes;
            this.interceptorContext = interceptorContext;
        }

        public CompletableFuture<TeaModel> execute() {
            CompletableFuture<TeaModel> future = new CompletableFuture<>();
            try {
                retryThenExecute(future);
            } catch (Exception e) {
                if (this.interceptorContext != null
                        && this.interceptorContext.httpResponseHandler() != null) {
                    this.interceptorContext.httpResponseHandler().onError(e);
                }
                future.completeExceptionally(e);
            }
            return future;
        }

        private void retryThenExecute(CompletableFuture<TeaModel> future) {
            attemptNumber++;

            //update RetryPolicyContext
            retryContext = RetryPolicyContext.builder()
                    .retriesAttempted(attemptNumber - 1)
                    .exception(lastException)
                    .httpResponse(lastHttpResponse)
                    .build();

            if (!needRetry()) {
                if (this.interceptorContext != null
                        && this.interceptorContext.httpResponseHandler() != null) {
                    this.interceptorContext.httpResponseHandler().onError(this.lastException);
                }
                future.completeExceptionally(lastException);
                return;
            }

            Duration backoffDelay = getBackoffDelay();
            if (!backoffDelay.isZero()) {
                //Todo active by other scheduled Executor
                try {
                    TimeUnit.SECONDS.sleep(backoffDelay.getSeconds());
                } catch (InterruptedException e) {
                    logger.error("Task defer failed, error message: {}", e.getMessage());
                }
            }
            attemptExecute(future);
        }

        private void retryThenExecute(CompletableFuture<TeaModel> future, Throwable exception, HttpResponse lastResponse) {
            lastException = exception;
            lastHttpResponse = lastResponse;
            retryThenExecute(future);
        }

        private boolean isFirstRequest() {
            return attemptNumber == 1 ? true : false;
        }

        private boolean needRetry() {
            if (isFirstRequest()) {
                return true;
            }

            if (retryPolicy == null) {
                return false;
            }

            if (retryPolicy.aggregateRetryCondition().shouldRetry(retryContext)) {
                return true;
            }
            return false;
        }

        private Duration getBackoffDelay() {
            Duration result;
            if (isFirstRequest()) {
                result = Duration.ZERO;
            } else {
                result = retryPolicy.backoffStrategy().computeDelayBeforeNextRetry(retryContext);
            }
            return result;
        }

        private void attemptExecute(CompletableFuture<TeaModel> future) {
            final InterceptorChain interceptorChain = configuration.option(ClientOption.INTERCEPTOR_CHAIN);
            CompletableFuture<HttpResponse> responseFuture = null;

            try {
                interceptorContext = interceptorChain.modifyHttpRequest(interceptorContext, attributes);
                responseFuture = httpClient.send(composeHttpRequest(interceptorContext),
                        composeHttpRequestContext(interceptorContext, attributes));
            } catch (ClientException | TeaException e) {
                retryThenExecute(future, e, null);
                return;
            } catch (Throwable e) {
                if (this.interceptorContext != null
                        && this.interceptorContext.httpResponseHandler() != null) {
                    this.interceptorContext.httpResponseHandler().onError(e);
                }
                future.completeExceptionally(e);
                return;
            }

            responseFuture.whenComplete((response, exception) -> {
                if (exception != null) {
                    retryThenExecute(future, exception, response);
                    return;
                }

                try {
                    //1. Cover to ServiceException
                    //2. pass to retry detection
                    interceptorContext.setHttpResponse(response);
                    interceptorContext = interceptorChain.modifyResponse(interceptorContext, attributes);
                    if (!CommonUtil.is2xx(response.getStatusCode()) ||
                            interceptorContext.teaResponse().exception() != null) {
                        retryThenExecute(future, interceptorContext.teaResponse().exception(), response);
                        return;
                    }

                    interceptorContext = interceptorChain.modifyOutput(interceptorContext, attributes);
                } catch (Throwable e) {
                    if (this.interceptorContext != null
                            && this.interceptorContext.httpResponseHandler() != null) {
                        this.interceptorContext.httpResponseHandler().onError(e);
                    }
                    future.completeExceptionally(e);
                    return;
                }
                future.complete(interceptorContext.output());
            });
        }
    }
}
