package darabonba.core.utils;

import com.aliyun.core.annotation.EnumType;
import com.aliyun.core.http.ProtocolType;
import com.aliyun.core.utils.StringUtils;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import darabonba.core.RequestModel;
import darabonba.core.exception.TeaException;
import darabonba.core.TeaModel;

import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URLEncoder;
import java.util.*;

import static com.aliyun.core.utils.EncodeUtil.encode;

public final class ModelUtil {
    private final static String UTF8 = "UTF-8";
    private final static String SEPARATOR = "&";

    public static Map<String, String> queryConvert(RequestModel model) {
        return query(model.getQueryParameters());
    }

    public static Object bodyConvert(RequestModel model, boolean bodyIsForm) {
        if (!bodyIsForm && !CommonUtil.isUnset(model.getBodyParameters()) && model.getBodyParameters().containsKey("body")) {
            return parseObject(model.getBodyParameters().get("body"));
        }
        return parseObject(model.getBodyParameters());
    }

    public static InputStream streamConvert(RequestModel model) {
        return parseStream(model.getBodyParameters());
    }

    public static Map<String, String> headersConvert(RequestModel model) {
        return parseHeaders(model.getHeaderParameters());
    }

    public static Map<String, String> hostConvert(RequestModel model) {
        return model.getHostParameters();
    }

    public static String pathConvert(RequestModel model, String pathRegex) {
        Map<String, String> pathMap = model.getPathParameters();
        String path = pathRegex;
        for (Map.Entry<String, String> entry : pathMap.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            if (value == null || "null".equals(value)) {
                value = "";
            }
            String regex = "\\{" + key + "\\}";
            try {
                path = path.replaceAll(regex, URLEncoder.encode(value, UTF8));
            } catch (UnsupportedEncodingException e) {
                throw new TeaException(e);
            }
        }
        return path;
    }

    /**
     * Parse filter into a object which's type is map[string]string
     *
     * @param filter query param
     * @return the object
     */
    public static Map<String, String> query(Map<String, ?> filter) {
        Map<String, String> outMap = new HashMap<>();
        if (!CommonUtil.isUnset(filter)) {
            processObject(outMap, "", filter);
        }
        return outMap;
    }

    private static void processObject(Map<String, String> map, String key, Object value) {
        if (CommonUtil.isUnset(value)) {
            return;
        }
        if (value instanceof List) {
            List list = (List) value;
            for (int i = 0; i < list.size(); i++) {
                processObject(map, key + "." + (i + 1), list.get(i));
            }
        } else if (value instanceof TeaModel) {
            Map<String, Object> subMap = ((TeaModel) value).toMap();
            for (Map.Entry<String, Object> entry : subMap.entrySet()) {
                processObject(map, key + "." + (entry.getKey()), entry.getValue());
            }
        } else if (value instanceof Map) {
            Map<String, Object> subMap = (Map<String, Object>) value;
            for (Map.Entry<String, Object> entry : subMap.entrySet()) {
                processObject(map, key + "." + (entry.getKey()), entry.getValue());
            }
        } else {
            if (key.startsWith(".")) {
                key = key.substring(1);
            }
            if (!CommonUtil.isUnset(value.getClass().getDeclaredAnnotation(EnumType.class))) {
                try {
                    Method method = value.getClass().getDeclaredMethod("getValue");
                    map.put(key, String.valueOf(method.invoke(value)));
                } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
                    throw new TeaException(e);
                }
            } else if (value instanceof byte[]) {
                try {
                    map.put(key, new String((byte[]) value, "UTF-8"));
                } catch (UnsupportedEncodingException e) {
                    throw new TeaException(e);
                }
            } else {
                map.put(key, String.valueOf(value));
            }
        }
    }

    public static Object parseObject(Object o) {
        if (CommonUtil.isUnset(o)) {
            return null;
        }
        Class clazz = o.getClass();
        if (List.class.isAssignableFrom(clazz)) {
            List<Object> list = (List<Object>) o;
            List<Object> result = new ArrayList<>();
            for (Object object : list) {
                result.add(parseObject(object));
            }
            return result;
        } else if (Map.class.isAssignableFrom(clazz)) {
            Map<String, Object> map = (Map<String, Object>) o;
            Map<String, Object> result = new HashMap<>();
            for (Map.Entry<String, Object> entry : map.entrySet()) {
                if (!CommonUtil.isUnset(entry.getValue()) && InputStream.class.isAssignableFrom(entry.getValue().getClass())) {
                    return null;
                }
                result.put(entry.getKey(), parseObject(entry.getValue()));
            }
            return result;
        } else if (TeaModel.class.isAssignableFrom(clazz)) {
            return ((TeaModel) o).toMap(false);
        } else {
            if (!CommonUtil.isUnset(clazz.getDeclaredAnnotation(EnumType.class))) {
                try {
                    Method method = clazz.getDeclaredMethod("getValue");
                    return method.invoke(o);
                } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
                    throw new TeaException(e);
                }
            }
            return o;
        }
    }

    public static InputStream parseStream(Map<String, Object> streamMap) {
        if (CommonUtil.isUnset(streamMap)) {
            return null;
        }
        for (Map.Entry<String, Object> entry : streamMap.entrySet()) {
            if (InputStream.class.isAssignableFrom(entry.getValue().getClass())) {
                return (InputStream) entry.getValue();
            }
        }
        return null;
    }

    public static Map<String, String> parseHeaders(Map<String, Object> headersMap) {
        // TODO deal more deep map
        Map<String, String> result = new HashMap<>();
        for (Map.Entry<String, Object> entry : headersMap.entrySet()) {
            if (Map.class.isAssignableFrom(entry.getValue().getClass())) {
                Map<String, Object> map = (Map<String, Object>) entry.getValue();
                String key = entry.getKey();
                for (Map.Entry<String, Object> item : map.entrySet()) {
                    result.put(key.replace("*", item.getKey()), String.valueOf(item.getValue()));
                }
            } else {
                if (!CommonUtil.isUnset(entry.getValue().getClass().getDeclaredAnnotation(EnumType.class))) {
                    try {
                        Method method = entry.getValue().getClass().getDeclaredMethod("getValue");
                        result.put(entry.getKey(), String.valueOf(method.invoke(entry.getValue())));
                    } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
                        throw new TeaException(e);
                    }
                } else if (entry.getValue() instanceof byte[]) {
                    try {
                        result.put(entry.getKey(), new String((byte[]) entry.getValue(), "UTF-8"));
                    } catch (UnsupportedEncodingException e) {
                        throw new TeaException(e);
                    }
                } else {
                    result.put(entry.getKey(), String.valueOf(entry.getValue()));
                }
            }
        }
        return result;
    }

    /**
     * Format a map to form string, like a=a%20b%20c
     *
     * @return the form string
     */
    public static String toFormString(Map<String, ?> map) {
        if (CommonUtil.isUnset(map)) {
            return null;
        }
        StringBuilder result = new StringBuilder();
        boolean first = true;
        try {
            for (Map.Entry<String, ?> entry : map.entrySet()) {
                if (StringUtils.isEmpty(entry.getValue())) {
                    continue;
                }
                if (first) {
                    first = false;
                } else {
                    result.append(SEPARATOR);
                }
                result.append(URLEncoder.encode(entry.getKey(), UTF8));
                result.append("=");
                result.append(URLEncoder.encode(String.valueOf(entry.getValue()), UTF8));
            }
        } catch (Exception e) {
            throw new TeaException(e.getMessage(), e);
        }
        return result.toString();
    }

    private static String toFormWithSymbol(java.util.Map<String, ?> filter, String symbol) {
        Map<String, String> map = query(filter);
        StringBuilder result = new StringBuilder();
        boolean first = true;
        try {
            for (Map.Entry<String, ?> entry : map.entrySet()) {
                if (StringUtils.isEmpty(entry.getValue())) {
                    continue;
                }
                if (first) {
                    first = false;
                } else {
                    result.append(symbol);
                }
                result.append(URLEncoder.encode(entry.getKey(), UTF8));
                result.append("=");
                result.append(URLEncoder.encode(String.valueOf(entry.getValue()), UTF8));
            }
        } catch (Exception e) {
            throw new TeaException(e.getMessage(), e);
        }
        return result.toString();
    }

    public static String composeUrl(String host, Map<String, String> query, String protocol, String pathname) {
        Map<String, String> queries = query;
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(StringUtils.isEmpty(protocol) ? ProtocolType.HTTPS : protocol);
        urlBuilder.append("://").append(host);
        if (!StringUtils.isEmpty(pathname)) {
            urlBuilder.append(pathname);
        }
        if (queries.size() > 0) {
            if (urlBuilder.indexOf("?") >= 1) {
                urlBuilder.append(SEPARATOR);
            } else {
                urlBuilder.append("?");
            }
            try {
                for (Map.Entry<String, String> entry : queries.entrySet()) {
                    String key = entry.getKey();
                    String val = entry.getValue();
                    if (null == val) {
                        continue;
                    }
                    urlBuilder.append(encode(key));
                    urlBuilder.append("=");
                    urlBuilder.append(encode(val));
                    urlBuilder.append(SEPARATOR);
                }
            } catch (Exception e) {
                throw new TeaException(e.getMessage(), e);
            }
            int strIndex = urlBuilder.length();
            urlBuilder.deleteCharAt(strIndex - 1);
        }
        return urlBuilder.toString();
    }

    public static String shrinkSpecifiedStyle(Object object, String prefix, String style) {
        if (null == object) {
            return null;
        }
        switch (style) {
            case "repeatList":
                Map<String, Object> map = new HashMap<String, Object>();
                map.put(prefix, object);
                return toFormWithSymbol(map, "&&");
            case "simple":
            case "spaceDelimited":
            case "pipeDelimited":
                return flatArray((List) object, style);
            case "json":
                Object newObject = parseObject(object);
                Gson gson = new GsonBuilder().disableHtmlEscaping().create();
                return gson.toJson(newObject);
            default:
                return "";
        }
    }

    private static String flatArray(List array, String sty) {
        if (array == null || array.size() <= 0 || sty == null) {
            return "";
        }
        String flag;
        if ("simple".equalsIgnoreCase(sty)) {
            flag = ",";
        } else if ("spaceDelimited".equalsIgnoreCase(sty)) {
            flag = " ";
        } else {
            flag = "|";
        }
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < array.size(); i++) {
            sb.append(array.get(i));
            sb.append(flag);
        }
        return sb.deleteCharAt(sb.length() - 1).toString();
    }
}
