/*
 * Decompiled with CFR 0.152.
 */
package com.caucho.db.block;

import com.caucho.db.Database;
import com.caucho.db.block.Block;
import com.caucho.db.block.BlockManager;
import com.caucho.db.block.BlockReadWrite;
import com.caucho.db.block.BlockWriter;
import com.caucho.env.health.HealthSystemFacade;
import com.caucho.lifecycle.Lifecycle;
import com.caucho.loader.Environment;
import com.caucho.util.L10N;
import com.caucho.vfs.Path;
import com.caucho.vfs.RandomAccessStream;
import java.io.IOException;
import java.io.OutputStream;
import java.sql.SQLException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Level;
import java.util.logging.Logger;

public class BlockStore {
    private static final Logger log = Logger.getLogger(BlockStore.class.getName());
    private static final L10N L = new L10N(BlockStore.class);
    public static final int BLOCK_BITS = 13;
    public static final int BLOCK_SIZE = 8192;
    public static final long BLOCK_INDEX_MASK = 8191L;
    public static final long BLOCK_MASK = -8192L;
    public static final long BLOCK_OFFSET_MASK = 8191L;
    private static final int ALLOC_BYTES_PER_BLOCK = 2;
    private static final int ALLOC_CHUNK_SIZE = 8192;
    private static final int ALLOC_GROUP_COUNT = 4096;
    private static final long ALLOC_GROUP_SIZE = 0x2000000L;
    public static final int ALLOC_FREE = 0;
    public static final int ALLOC_ROW = 1;
    public static final int ALLOC_DATA = 2;
    public static final int ALLOC_INODE_PTR = 3;
    public static final int ALLOC_INDEX = 4;
    public static final int ALLOC_MINI_FRAG = 5;
    public static final int ALLOC_MASK = 15;
    public static final int MINI_FRAG_SIZE = 256;
    public static final int MINI_FRAG_PER_BLOCK = 31;
    public static final int MINI_FRAG_ALLOC_OFFSET = 7936;
    private static final int MINI_FRAG_FREE_STRIDE = 4;
    private static final int MINI_FRAG_STRIDE_MASK = 3;
    public static final long METADATA_START = 8192L;
    public static final int STORE_CREATE_END = 1024;
    public static final String DATABASE_CORRUPT_EVENT = "caucho.database.corrupt";
    private final String _name;
    private final Path _path;
    private int _id;
    protected final Database _database;
    protected final BlockManager _blockManager;
    private final BlockReadWrite _readWrite;
    private final BlockWriter _writer;
    private boolean _isFlushDirtyBlocksOnCommit = true;
    private long _blockCount;
    private final Object _allocationLock = new Object();
    private byte[] _allocationTable;
    private long _freeAllocIndex;
    private int _freeAllocCount;
    private int _freeMiniAllocIndex;
    private int _freeMiniAllocCount;
    private final AtomicLong _freeMiniOffset = new AtomicLong();
    private final Object _allocationWriteLock = new Object();
    private final AtomicInteger _allocationWriteCount = new AtomicInteger();
    private int _allocDirtyMin = Integer.MAX_VALUE;
    private int _allocDirtyMax;
    private long _miniFragmentUseCount;
    private Lock _rowWriteLock;
    private long _blockLockTimeout = 120000L;
    private final Lifecycle _lifecycle = new Lifecycle();
    private AtomicInteger _allocCount = new AtomicInteger();

    public BlockStore(Database database, String name, ReadWriteLock tableLock) {
        this(database, name, tableLock, database.getPath().lookup(name + ".db"));
    }

    public BlockStore(Database database, String name, ReadWriteLock rowLock, Path path) {
        this(database, name, rowLock, path, BlockManager.getBlockManager().isEnableMmap());
    }

    public BlockStore(Database database, String name, ReadWriteLock rowLock, Path path, boolean isEnableMmap) {
        this._database = database;
        this._blockManager = this._database.getBlockManager();
        this._name = name;
        this._id = this._blockManager.allocateStoreId();
        if (path == null) {
            throw new NullPointerException();
        }
        this._path = path;
        String exitMessage = HealthSystemFacade.getExitMessage();
        if (exitMessage.indexOf(path.getFullPath()) >= 0) {
            log.warning("removing " + this._path.getFullPath() + " due to restart corruption");
            try {
                this._path.remove();
            }
            catch (Exception e) {
                log.log(Level.FINE, e.toString(), e);
            }
        }
        this._readWrite = new BlockReadWrite(this, path, isEnableMmap);
        this._writer = new BlockWriter(this);
        if (rowLock == null) {
            rowLock = new ReentrantReadWriteLock();
        }
        rowLock.readLock();
        this._rowWriteLock = rowLock.writeLock();
        Environment.addCloseListener(this);
    }

    public static BlockStore create(Path path) throws IOException, SQLException {
        return BlockStore.create(path, true);
    }

    public static BlockStore createNoMmap(Path path) throws IOException, SQLException {
        return BlockStore.create(path, false);
    }

    public static BlockStore createMmap(Path path) throws IOException, SQLException {
        return BlockStore.create(path, true);
    }

    public static BlockStore create(Path path, boolean isMmap) throws IOException, SQLException {
        Database db = new Database();
        db.init();
        BlockStore store = new BlockStore(db, "temp", null, path, isMmap);
        if (path.canRead()) {
            store.init();
        } else {
            store.create();
        }
        return store;
    }

    public void setEnableMmap(boolean isEnable) {
    }

    public void setFlushDirtyBlocksOnCommit(boolean flushOnCommit) {
        this._isFlushDirtyBlocksOnCommit = flushOnCommit;
    }

    public boolean isFlushDirtyBlocksOnCommit() {
        return this._isFlushDirtyBlocksOnCommit;
    }

    public String getName() {
        return this._name;
    }

    public int getId() {
        return this._id;
    }

    public Path getPath() {
        return this._path;
    }

    public Lock getWriteLock() {
        return this._rowWriteLock;
    }

    public Lock getTableLock() {
        return this._rowWriteLock;
    }

    public BlockManager getBlockManager() {
        return this._blockManager;
    }

    protected BlockReadWrite getReadWrite() {
        return this._readWrite;
    }

    BlockWriter getWriter() {
        return this._writer;
    }

    public RandomAccessStream getMmap() {
        return this._readWrite.getMmap();
    }

    public long getFileSize() {
        return this._readWrite.getFileSize();
    }

    public long getBlockCount() {
        return this._blockCount;
    }

    public static long blockIndexToAddr(long blockIndex) {
        return blockIndex << 13;
    }

    private final long blockIndexToBlockId(long blockIndex) {
        return (blockIndex << 13) + (long)this._id;
    }

    public static long blockIdToIndex(long blockId) {
        return blockId >> 13;
    }

    public final long addressToBlockId(long address) {
        return (address & 0xFFFFFFFFFFFFE000L) + (long)this._id;
    }

    public static long blockIdToAddress(long blockId) {
        return blockId & 0xFFFFFFFFFFFFE000L;
    }

    public static long blockIdToAddress(long blockId, int offset) {
        return (blockId & 0xFFFFFFFFFFFFE000L) + (long)offset;
    }

    public void create() throws IOException, SQLException {
        if (!this._lifecycle.toActive()) {
            return;
        }
        log.finer(this + " create");
        this._readWrite.create();
        this._allocationTable = new byte[8192];
        this.setAllocation(0L, 2);
        this.setAllocation(1L, 2);
        boolean isPriority = true;
        byte[] buffer = new byte[8192];
        this._readWrite.writeBlock(0L, this._allocationTable, 0, this._allocationTable.length, isPriority);
        this._readWrite.writeBlock(8192L, buffer, 0, 8192, isPriority);
        this._blockCount = 2L;
        if (this.getAllocation(0L) != 2 || this.getAllocation(1L) != 2) {
            Thread.dumpStack();
        }
    }

    public void init() throws IOException {
        if (!this._lifecycle.toActive()) {
            return;
        }
        log.finer(this + " init");
        this._readWrite.init();
        this._blockCount = (this.getFileSize() + 8192L - 1L) / 8192L;
        int allocCount = (int)this._blockCount;
        allocCount += 4095;
        allocCount -= allocCount % 4096;
        int allocSize = allocCount * 2;
        if (allocSize < 8192) {
            log.warning(this + " chunk failure. Rebuilding.");
            this.removeAndCreate();
            return;
        }
        this._allocationTable = new byte[allocSize];
        for (int i = 0; i < allocSize; i += 8192) {
            long allocGroup = i / 8192;
            this._readWrite.readBlock(allocGroup * 0x2000000L, this._allocationTable, i, 8192);
        }
        if (!this.validateLoad()) {
            this.removeAndCreate();
        }
    }

    private boolean validateLoad() {
        if (this.getAllocation(0L) != 2 || this.getAllocation(1L) != 2) {
            log.warning(this + " corrupted block=zero database. Rebuilding.");
            Thread.dumpStack();
            return false;
        }
        long superBlockMax = this._allocationTable.length / 2;
        for (long index = 0L; index < superBlockMax; index += 4096L) {
            if (this.getAllocation(index) == 2) continue;
            log.warning(L.l(this + " corrupted database meta-data {0} for address=0x{1}. Rebuilding.", (Object)this.getAllocation(index), (Object)Long.toHexString(index * 8192L)));
            Thread.dumpStack();
            return false;
        }
        return true;
    }

    private void removeAndCreate() {
        if (!this._lifecycle.toIdle()) {
            Thread.dumpStack();
        }
        try {
            this._readWrite.removeInit();
        }
        catch (Exception e) {
            e.printStackTrace();
        }
        try {
            this.create();
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void remove() throws SQLException {
        this._readWrite.remove();
        this.close();
    }

    public long firstRowBlock(long blockId) throws IOException {
        return this.firstBlock(blockId, 1);
    }

    public long firstBlock(long blockId, int type) throws IOException {
        if (blockId <= 8192L) {
            blockId = 8192L;
        }
        long blockCount = this._blockCount;
        for (long blockIndex = blockId >> 13; blockIndex < blockCount; ++blockIndex) {
            if (this.getAllocation(blockIndex) != type) continue;
            return this.blockIndexToBlockId(blockIndex);
        }
        return -1L;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final Block readBlock(long blockAddress) throws IOException {
        long blockId = this.addressToBlockId(blockAddress);
        Block block = this._blockManager.getBlock(this, blockId);
        boolean isValid = false;
        try {
            block.read();
            isValid = true;
            Block block2 = block;
            return block2;
        }
        finally {
            if (!isValid) {
                block.free();
            }
        }
    }

    public final Block loadBlock(long blockAddress) throws IOException {
        long blockId = this.addressToBlockId(blockAddress);
        Block block = this._blockManager.getBlock(this, blockId);
        return block;
    }

    public Block allocateRow() throws IOException {
        boolean isSave = true;
        Block block = this.allocateBlock(1, isSave);
        return block;
    }

    public boolean isRowBlock(long blockAddress) {
        return this.getAllocationByAddress(blockAddress) == 1;
    }

    public boolean isMiniFragBlock(long blockAddress) {
        return this.getAllocationByAddress(blockAddress) == 5;
    }

    public Block allocateBlock() throws IOException {
        boolean isSave = true;
        return this.allocateBlock(2, isSave);
    }

    public Block allocateIndirectBlock() throws IOException {
        boolean isSave = true;
        return this.allocateBlock(3, isSave);
    }

    private Block allocateBlockMiniFragment() throws IOException {
        boolean isSave = true;
        return this.allocateBlock(5, isSave);
    }

    public Block allocateIndexBlock() throws IOException {
        boolean isSave = false;
        return this.allocateBlock(4, isSave);
    }

    public boolean isIndexBlock(long blockAddress) {
        return this.getAllocationByAddress(blockAddress) == 4;
    }

    public boolean isInodePtrBlock(long blockAddress) {
        return this.getAllocationByAddress(blockAddress) == 3;
    }

    public boolean isDataBlock(long blockAddress) {
        return this.getAllocationByAddress(blockAddress) == 2;
    }

    private Block allocateBlock(int code, boolean isSave) throws IOException {
        long blockIndex;
        while ((blockIndex = this.findFreeBlock(code)) == 0L) {
            if (this._freeAllocIndex != this._blockCount || this._freeAllocCount != 0) continue;
            this.extendFile();
        }
        long blockId = this.blockIndexToBlockId(blockIndex);
        Block block = this._blockManager.getBlock(this, blockId);
        byte[] buffer = block.getBuffer();
        for (int i = 8191; i >= 0; --i) {
            buffer[i] = 0;
        }
        block.setDirty(0, 8192);
        block.toValid();
        this._allocCount.incrementAndGet();
        this.saveAllocation();
        return block;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long findFreeBlock(int code) {
        if (code == 0) {
            throw new IllegalStateException();
        }
        Object object = this._allocationLock;
        synchronized (object) {
            long end = this._blockCount;
            if ((long)this._allocationTable.length < 2L * end) {
                end = this._allocationTable.length / 2;
            }
            for (long blockIndex = this._freeAllocIndex; blockIndex < end; ++blockIndex) {
                if (this.getAllocation(blockIndex) != 0) continue;
                this._freeAllocIndex = blockIndex;
                ++this._freeAllocCount;
                this.setAllocation(blockIndex, code);
                return blockIndex;
            }
            if (this._freeAllocCount > 0) {
                this._freeAllocIndex = 0L;
                this._freeAllocCount = 0;
            } else {
                this._freeAllocIndex = this._blockCount;
            }
            return 0L;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void extendFile() {
        Object object = this._allocationLock;
        synchronized (object) {
            if (this._freeAllocIndex < this._blockCount) {
                return;
            }
            long newBlockCount = this._blockCount < 256L ? this._blockCount + 1L : this._blockCount + 256L;
            newBlockCount = Math.max(newBlockCount, this._readWrite.getFileSize() / 8192L);
            while ((long)(this._allocationTable.length / 2) < newBlockCount) {
                byte[] newTable = new byte[this._allocationTable.length + 8192];
                System.arraycopy(this._allocationTable, 0, newTable, 0, this._allocationTable.length);
                this._allocationTable = newTable;
                if (this.getAllocation(0L) != 2 || this.getAllocation(1L) != 2) {
                    Thread.dumpStack();
                }
                long superBlockMax = this._allocationTable.length / 2;
                for (long index = 0L; index < superBlockMax; index += 4096L) {
                    this.setAllocation(index, 2);
                    if (newBlockCount != index + 1L) continue;
                    ++newBlockCount;
                }
                this.setAllocDirty(0, newTable.length);
            }
            if (log.isLoggable(Level.FINER)) {
                log.finer(this + " extending file " + newBlockCount);
            }
            this._blockCount = newBlockCount;
            this._freeAllocIndex = 0L;
            long newBlockIndex = newBlockCount - 1L;
            if (this.getAllocation(newBlockIndex) != 0) {
                System.out.println(this + " BAD_BLOCK: " + newBlockIndex + " " + this.getAllocation(newBlockIndex));
            }
            this.setAllocation(newBlockIndex, 2);
            long blockId = this.blockIndexToBlockId(newBlockIndex);
            Block block = this._blockManager.getBlock(this, blockId);
            byte[] buffer = block.getBuffer();
            for (int i = 8191; i >= 0; --i) {
                buffer[i] = 0;
            }
            block.toValid();
            block.setDirty(0, 8192);
            try {
                block.writeFromBlockWriter();
            }
            catch (IOException e) {
                log.log(Level.WARNING, e.toString(), e);
            }
            block.free();
        }
        try {
            this.saveAllocation();
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void validateBlockId(long blockId) throws IllegalArgumentException, IllegalStateException {
        RuntimeException e = null;
        if (this.isClosed()) {
            e = new IllegalStateException(L.l("store {0} is closing.", (Object)this));
        } else if (this.getId() <= 0) {
            e = new IllegalStateException(L.l("invalid store {0}.", (Object)this));
        } else if ((long)this.getId() != (blockId & 0x1FFFL)) {
            e = new IllegalArgumentException(L.l("block 0x{0} index {1} must match store {2}.", (Object)Long.toHexString(blockId), (Object)(blockId & 0x1FFFL), (Object)this));
        } else if (BlockStore.blockIdToAddress(blockId) <= 0L) {
            e = new IllegalArgumentException(L.l("invalid block address 0x{0} for store {1}.", (Object)Long.toHexString(blockId), (Object)this));
        }
        if (e != null) {
            throw e;
        }
    }

    protected void assertStoreActive() throws IllegalStateException {
        IllegalStateException e = null;
        if (this.isClosed()) {
            e = new IllegalStateException(L.l("store {0} is closing.", (Object)this));
        } else if (this.getId() <= 0) {
            e = new IllegalStateException(L.l("invalid store {0}.", (Object)this));
        }
        if (e != null) {
            throw e;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deallocateBlock(long blockId) throws IOException {
        if (blockId <= 1L) {
            Thread.dumpStack();
            return;
        }
        long index = BlockStore.blockIdToIndex(blockId);
        Object object = this._allocationLock;
        synchronized (object) {
            if (this.getAllocation(index) == 0) {
                throw new IllegalStateException(L.l("{0} double free of {1}", (Object)this, (Object)Long.toHexString(blockId)));
            }
            this.setAllocation(index, 0);
            this._allocCount.decrementAndGet();
        }
        this.saveAllocation();
    }

    public final int getAllocationByAddress(long blockAddress) {
        return this.getAllocation(blockAddress / 8192L);
    }

    public final int getAllocation(long blockIndex) {
        int allocOffset = (int)(2L * blockIndex);
        if (allocOffset < 0 || this._allocationTable.length <= allocOffset) {
            return -1;
        }
        return this._allocationTable[allocOffset] & 0xF;
    }

    private void setAllocation(long blockIndex, int code) {
        if (blockIndex <= 1L && code != 2) {
            System.out.println("Suspicious change: 0x" + Long.toHexString(blockIndex) + " " + code);
            Thread.dumpStack();
            return;
        }
        if (blockIndex % 4096L == 0L && code != 2) {
            System.out.println("Suspicious meta-data: 0x" + Long.toHexString(blockIndex) + " " + code);
            Thread.dumpStack();
            return;
        }
        int allocOffset = (int)(2L * blockIndex);
        for (int i = 1; i < 2; ++i) {
            this._allocationTable[allocOffset + i] = 0;
        }
        int oldCode = this._allocationTable[allocOffset] & 0xF;
        this._allocationTable[allocOffset] = (byte)code;
        if (oldCode != 0 && code != 0 && oldCode != code) {
            System.out.println("Suspicious change: " + Long.toHexString(blockIndex) + " old:" + oldCode + " new:" + code);
            Thread.dumpStack();
        }
        this.setAllocDirty(allocOffset, allocOffset + 2);
    }

    private void setAllocDirty(int min, int max) {
        this._allocDirtyMin = Math.min(min, this._allocDirtyMin);
        this._allocDirtyMax = Math.max(max, this._allocDirtyMax);
    }

    public void saveAllocation() throws IOException {
        if (!this._isFlushDirtyBlocksOnCommit) {
            return;
        }
        while (this._allocDirtyMin < this._allocDirtyMax) {
            try {
                if (this._allocationWriteCount.getAndIncrement() > 2) {
                    return;
                }
                this.writeAllocation();
            }
            finally {
                this._allocationWriteCount.decrementAndGet();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeAllocation() throws IOException {
        Object object = this._allocationWriteLock;
        synchronized (object) {
            int dirtyMax;
            int dirtyMin;
            Object object2 = this._allocationLock;
            synchronized (object2) {
                dirtyMin = this._allocDirtyMin;
                this._allocDirtyMin = Integer.MAX_VALUE;
                dirtyMax = this._allocDirtyMax;
                this._allocDirtyMax = 0;
            }
            this.saveAllocation(dirtyMin, dirtyMax);
        }
    }

    private void saveAllocation(int dirtyMin, int dirtyMax) throws IOException {
        while (dirtyMin < dirtyMax) {
            int allocGroup = dirtyMin / 8192;
            int offset = dirtyMin % 8192;
            int length = dirtyMin / 8192 != dirtyMax / 8192 ? 8192 - offset : dirtyMax - dirtyMin;
            boolean isPriority = true;
            this._readWrite.writeBlock((long)allocGroup * 0x2000000L + (long)offset, this._allocationTable, dirtyMin, length, isPriority);
            dirtyMin += length;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void readBlock(long blockId, int blockOffset, OutputStream os, int length) throws IOException {
        if (blockId <= 0L) {
            log.warning(this + " illegal block read with block-id=0");
            return;
        }
        if (8192 - blockOffset < length) {
            throw new IllegalArgumentException(L.l("read offset {0} length {1} too long", blockOffset, length));
        }
        Block block = this.readBlock(blockId);
        try {
            Lock lock = block.getReadLock();
            lock.tryLock(this._blockLockTimeout, TimeUnit.MILLISECONDS);
            try {
                byte[] blockBuffer = block.getBuffer();
                os.write(blockBuffer, blockOffset, length);
            }
            finally {
                lock.unlock();
            }
        }
        catch (InterruptedException e) {
            throw new IllegalStateException(e);
        }
        finally {
            block.free();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void readBlockNoLock(long blockId, int blockOffset, OutputStream os, int length) throws IOException {
        if (blockId <= 0L) {
            log.warning(this + " illegal block read with block-id=0");
            return;
        }
        if (8192 - blockOffset < length) {
            throw new IllegalArgumentException(L.l("read offset {0} length {1} too long", blockOffset, length));
        }
        Block block = this.readBlock(blockId);
        try {
            byte[] blockBuffer = block.getBuffer();
            os.write(blockBuffer, blockOffset, length);
        }
        finally {
            block.free();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int readBlock(long blockAddress, int blockOffset, byte[] buffer, int offset, int length) throws IOException {
        if (8192 - blockOffset < length) {
            throw new IllegalArgumentException(L.l("read offset {0} length {1} too long", blockOffset, length));
        }
        Block block = this.readBlock(this.addressToBlockId(blockAddress));
        try {
            Lock lock = block.getReadLock();
            lock.tryLock(this._blockLockTimeout, TimeUnit.MILLISECONDS);
            try {
                byte[] blockBuffer = block.getBuffer();
                System.arraycopy(blockBuffer, blockOffset, buffer, offset, length);
                int n = length;
                lock.unlock();
                return n;
            }
            catch (Throwable throwable) {
                try {
                    lock.unlock();
                    throw throwable;
                }
                catch (InterruptedException e) {
                    throw new IllegalStateException(e);
                }
            }
        }
        finally {
            block.free();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int readBlock(long blockAddress, int blockOffset, char[] buffer, int offset, int length) throws IOException {
        if (8192 - blockOffset < 2 * length) {
            throw new IllegalArgumentException(L.l("read offset {0} length {1} too long", blockOffset, length));
        }
        Block block = this.readBlock(this.addressToBlockId(blockAddress));
        try {
            Lock lock = block.getReadLock();
            lock.tryLock(this._blockLockTimeout, TimeUnit.MILLISECONDS);
            try {
                byte[] blockBuffer = block.getBuffer();
                for (int i = 0; i < length; ++i) {
                    int ch1 = blockBuffer[blockOffset] & 0xFF;
                    int ch2 = blockBuffer[blockOffset + 1] & 0xFF;
                    buffer[offset + i] = (char)((ch1 << 8) + ch2);
                    blockOffset += 2;
                }
                int n = length;
                lock.unlock();
                return n;
            }
            catch (Throwable throwable) {
                try {
                    lock.unlock();
                    throw throwable;
                }
                catch (InterruptedException e) {
                    throw new IllegalStateException(e);
                }
            }
        }
        finally {
            block.free();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long readBlockLong(long blockAddress, int offset) throws IOException {
        Block block = this.readBlock(this.addressToBlockId(blockAddress));
        try {
            Lock lock = block.getReadLock();
            lock.tryLock(this._blockLockTimeout, TimeUnit.MILLISECONDS);
            try {
                byte[] blockBuffer = block.getBuffer();
                long l = BlockStore.readLong(blockBuffer, offset);
                lock.unlock();
                return l;
            }
            catch (Throwable throwable) {
                try {
                    lock.unlock();
                    throw throwable;
                }
                catch (InterruptedException e) {
                    throw new IllegalStateException(e);
                }
            }
        }
        finally {
            block.free();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Block writeBlock(long blockAddress, int blockOffset, byte[] buffer, int offset, int length) throws IOException {
        if (8192 - blockOffset < length) {
            throw new IllegalArgumentException(L.l("write offset {0} length {1} too long", blockOffset, length));
        }
        Block block = this.readBlock(this.addressToBlockId(blockAddress));
        try {
            Lock lock = block.getWriteLock();
            lock.tryLock(this._blockLockTimeout, TimeUnit.MILLISECONDS);
            try {
                byte[] blockBuffer = block.getBuffer();
                System.arraycopy(buffer, offset, blockBuffer, blockOffset, length);
                block.setDirty(blockOffset, blockOffset + length);
                Block block2 = block;
                lock.unlock();
                return block2;
            }
            catch (Throwable throwable) {
                try {
                    lock.unlock();
                    throw throwable;
                }
                catch (InterruptedException e) {
                    throw new IllegalStateException(e);
                }
            }
        }
        finally {
            block.free();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Block writeBlock(long blockAddress, int blockOffset, char[] buffer, int offset, int charLength) throws IOException {
        int length = 2 * charLength;
        if (8192 - blockOffset < length) {
            throw new IllegalArgumentException(L.l("write offset {0} length {1} too long", blockOffset, length));
        }
        Block block = this.readBlock(this.addressToBlockId(blockAddress));
        try {
            Lock lock = block.getWriteLock();
            lock.tryLock(this._blockLockTimeout, TimeUnit.MILLISECONDS);
            try {
                byte[] blockBuffer = block.getBuffer();
                int blockTail = blockOffset;
                for (int i = 0; i < charLength; ++i) {
                    char ch = buffer[offset + i];
                    blockBuffer[blockTail] = (byte)(ch >> 8);
                    blockBuffer[blockTail + 1] = (byte)ch;
                    blockTail += 2;
                }
                block.setDirty(blockOffset, blockTail);
                Block block2 = block;
                lock.unlock();
                return block2;
            }
            catch (Throwable throwable) {
                try {
                    lock.unlock();
                    throw throwable;
                }
                catch (InterruptedException e) {
                    throw new IllegalStateException(e);
                }
            }
        }
        finally {
            block.free();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Block writeBlockLong(long blockAddress, int offset, long value) throws IOException {
        Block block = this.readBlock(this.addressToBlockId(blockAddress));
        try {
            Lock lock = block.getWriteLock();
            lock.tryLock(this._blockLockTimeout, TimeUnit.MILLISECONDS);
            try {
                byte[] blockBuffer = block.getBuffer();
                BlockStore.writeLong(blockBuffer, offset, value);
                block.setDirty(offset, offset + 8);
                Block block2 = block;
                lock.unlock();
                return block2;
            }
            catch (Throwable throwable) {
                try {
                    lock.unlock();
                    throw throwable;
                }
                catch (InterruptedException e) {
                    throw new IllegalStateException(e);
                }
            }
        }
        finally {
            block.free();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int readMiniFragment(long fragmentAddress, int fragmentOffset, byte[] buffer, int offset, int length) throws IOException {
        if (256 - fragmentOffset < length) {
            throw new IllegalArgumentException(L.l("read offset {0} length {1} too long", fragmentOffset, length));
        }
        Block block = this.readBlock(this.addressToBlockId(fragmentAddress));
        try {
            Lock lock = block.getReadLock();
            lock.tryLock(this._blockLockTimeout, TimeUnit.MILLISECONDS);
            try {
                int blockOffset = this.getMiniFragmentOffset(fragmentAddress);
                byte[] blockBuffer = block.getBuffer();
                System.arraycopy(blockBuffer, blockOffset + fragmentOffset, buffer, offset, length);
                int n = length;
                lock.unlock();
                return n;
            }
            catch (Throwable throwable) {
                try {
                    lock.unlock();
                    throw throwable;
                }
                catch (InterruptedException e) {
                    throw new IllegalStateException(e);
                }
            }
        }
        finally {
            block.free();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int readMiniFragmentNoLock(long fragmentAddress, int fragmentOffset, int length, OutputStream os) throws IOException {
        if (256 - fragmentOffset < length) {
            throw new IllegalArgumentException(L.l("read offset {0} length {1} too long", fragmentOffset, length));
        }
        Block block = this.readBlock(this.addressToBlockId(fragmentAddress));
        try {
            int blockOffset = this.getMiniFragmentOffset(fragmentAddress);
            byte[] blockBuffer = block.getBuffer();
            os.write(blockBuffer, blockOffset + fragmentOffset, length);
            int n = length;
            return n;
        }
        finally {
            block.free();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int readMiniFragment(long fragmentAddress, int fragmentOffset, char[] buffer, int offset, int length) throws IOException {
        if (256 - fragmentOffset < 2 * length) {
            throw new IllegalArgumentException(L.l("read offset {0} length {1} too long", fragmentOffset, length));
        }
        Block block = this.readBlock(this.addressToBlockId(fragmentAddress));
        try {
            Lock lock = block.getReadLock();
            lock.tryLock(this._blockLockTimeout, TimeUnit.MILLISECONDS);
            try {
                int blockOffset = this.getMiniFragmentOffset(fragmentAddress);
                blockOffset += fragmentOffset;
                byte[] blockBuffer = block.getBuffer();
                for (int i = 0; i < length; ++i) {
                    int ch1 = blockBuffer[blockOffset] & 0xFF;
                    int ch2 = blockBuffer[blockOffset + 1] & 0xFF;
                    buffer[offset + i] = (char)((ch1 << 8) + ch2);
                    blockOffset += 2;
                }
                int n = length;
                lock.unlock();
                return n;
            }
            catch (Throwable throwable) {
                try {
                    lock.unlock();
                    throw throwable;
                }
                catch (InterruptedException e) {
                    throw new IllegalStateException(e);
                }
            }
        }
        finally {
            block.free();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long readMiniFragmentLong(long fragmentAddress, int fragmentOffset) throws IOException {
        Block block = this.readBlock(this.addressToBlockId(fragmentAddress));
        try {
            Lock lock = block.getReadLock();
            lock.tryLock(this._blockLockTimeout, TimeUnit.MILLISECONDS);
            try {
                int blockOffset = this.getMiniFragmentOffset(fragmentAddress);
                byte[] blockBuffer = block.getBuffer();
                long l = BlockStore.readLong(blockBuffer, blockOffset + fragmentOffset);
                lock.unlock();
                return l;
            }
            catch (Throwable throwable) {
                try {
                    lock.unlock();
                    throw throwable;
                }
                catch (InterruptedException e) {
                    throw new IllegalStateException(e);
                }
            }
        }
        finally {
            block.free();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long allocateMiniFragment() throws IOException {
        while (true) {
            long blockAddr = this.allocateMiniFragmentBlock();
            Block block = this.readBlock(blockAddr);
            int fragOffset = -1;
            try {
                byte[] blockBuffer = block.getBuffer();
                int freeOffset = -1;
                Lock lock = block.getWriteLock();
                lock.tryLock(this._blockLockTimeout, TimeUnit.MILLISECONDS);
                try {
                    int mask;
                    int offset;
                    int i;
                    for (i = 0; i < 31; ++i) {
                        offset = i / 8 + 7936;
                        mask = 1 << i % 8;
                        if ((blockBuffer[offset] & mask) != 0) continue;
                        fragOffset = i;
                        int n = offset;
                        blockBuffer[n] = (byte)(blockBuffer[n] | mask);
                        block.setDirty(offset, offset + 1);
                        break;
                    }
                    if (fragOffset < 0) continue;
                    for (i = 0; i < 31; ++i) {
                        offset = i / 8 + 7936;
                        mask = 1 << i % 8;
                        if ((blockBuffer[offset] & mask) != 0) continue;
                        freeOffset = (int)(2L * (blockAddr / 8192L));
                        break;
                    }
                }
                finally {
                    lock.unlock();
                    continue;
                }
                if (freeOffset >= 0) {
                    Object object = this._allocationLock;
                    synchronized (object) {
                        this._allocationTable[freeOffset + 1] = 0;
                        this.setAllocDirty(freeOffset + 1, freeOffset + 2);
                    }
                }
                long l = blockAddr + (long)fragOffset;
                return l;
            }
            catch (InterruptedException e) {
                throw new IllegalStateException(e);
            }
            finally {
                block.free();
                continue;
            }
            break;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long allocateMiniFragmentBlock() throws IOException {
        int offsetStride = 4;
        int offsetMask = 3;
        int allocStride = 2 * offsetStride;
        while (true) {
            byte[] allocationTable = this._allocationTable;
            int offset = (int)(this._freeMiniOffset.getAndIncrement() & (long)offsetMask);
            for (int i = this._freeMiniAllocIndex + offset * 2; i < allocationTable.length; i += allocStride) {
                int fragMask = allocationTable[i + 1] & 0xFF;
                if (allocationTable[i] != 5 || fragMask == 255) continue;
                this.updateFreeMiniAllocIndex(i);
                ++this._freeMiniAllocCount;
                Object object = this._allocationLock;
                synchronized (object) {
                    if (allocationTable[i] == 5 && fragMask != 255) {
                        allocationTable[i + 1] = -1;
                        this.setAllocDirty(i + 1, i + 2);
                        ++this._miniFragmentUseCount;
                        long fragmentAddress = 8192L * ((long)i / 2L);
                        return fragmentAddress;
                    }
                    continue;
                }
            }
            if (this._freeMiniAllocCount == 0) {
                int count = 32;
                for (int i = 0; i < count; ++i) {
                    Block block = this.allocateBlockMiniFragment();
                    block.free();
                }
            }
            this._freeMiniAllocCount = 0;
            this._freeMiniAllocIndex = 0;
        }
    }

    private void updateFreeMiniAllocIndex(int i) {
        byte[] allocationTable = this._allocationTable;
        int offset = this._freeMiniAllocIndex;
        if (offset == (i & 0xFFFFFFFC)) {
            return;
        }
        for (int j = 0; j < 4; ++j) {
            byte code = allocationTable[offset];
            int fragMask = allocationTable[offset + 1] & 0xFF;
            if (code == 5 && fragMask != 255) {
                return;
            }
            offset += 2;
        }
        this._freeMiniAllocIndex = i & 0xFFFFFFFC;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteMiniFragment(long fragmentAddress) throws IOException {
        Block block = this.readBlock(fragmentAddress);
        try {
            Lock lock = block.getWriteLock();
            lock.tryLock(this._blockLockTimeout, TimeUnit.MILLISECONDS);
            try {
                int fragIndex = (int)(fragmentAddress & 0x1FFFL);
                int offset = fragIndex / 8 + 7936;
                int mask = 1 << fragIndex % 8;
                byte[] blockBuffer = block.getBuffer();
                int n = offset;
                blockBuffer[n] = (byte)(blockBuffer[n] & ~mask);
                block.setDirty(offset, offset + 1);
                int i = (int)(2L * (fragmentAddress / 8192L));
                Object object = this._allocationLock;
                synchronized (object) {
                    int fragMask = this._allocationTable[i + 1] & 0xFF;
                    if (this._allocationTable[i] != 5) {
                        System.out.println("BAD ENTRY: " + fragMask);
                    }
                    this._allocationTable[i + 1] = 0;
                    --this._miniFragmentUseCount;
                    this.setAllocDirty(i + 1, i + 2);
                }
            }
            finally {
                lock.unlock();
            }
        }
        catch (InterruptedException e) {
            throw new IllegalStateException(e);
        }
        finally {
            block.free();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Block writeMiniFragment(long fragmentAddress, int fragmentOffset, byte[] buffer, int offset, int length) throws IOException {
        if (256 - fragmentOffset < length) {
            throw new IllegalArgumentException(L.l("write offset {0} length {1} too long", fragmentOffset, length));
        }
        Block block = this.readBlock(this.addressToBlockId(fragmentAddress));
        try {
            Lock lock = block.getWriteLock();
            lock.tryLock(this._blockLockTimeout, TimeUnit.MILLISECONDS);
            try {
                int blockOffset = this.getMiniFragmentOffset(fragmentAddress);
                byte[] blockBuffer = block.getBuffer();
                System.arraycopy(buffer, offset, blockBuffer, blockOffset += fragmentOffset, length);
                block.setDirty(blockOffset, blockOffset + length);
                Block block2 = block;
                lock.unlock();
                return block2;
            }
            catch (Throwable throwable) {
                try {
                    lock.unlock();
                    throw throwable;
                }
                catch (InterruptedException e) {
                    throw new IllegalStateException(e);
                }
            }
        }
        finally {
            block.free();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Block writeMiniFragment(long fragmentAddress, int fragmentOffset, char[] buffer, int offset, int length) throws IOException {
        if (256 - fragmentOffset < length) {
            throw new IllegalArgumentException(L.l("write offset {0} length {1} too long", fragmentOffset, length));
        }
        Block block = this.readBlock(this.addressToBlockId(fragmentAddress));
        try {
            Lock lock = block.getWriteLock();
            lock.tryLock(this._blockLockTimeout, TimeUnit.MILLISECONDS);
            try {
                int blockOffset = this.getMiniFragmentOffset(fragmentAddress);
                byte[] blockBuffer = block.getBuffer();
                int blockTail = blockOffset += fragmentOffset;
                for (int i = 0; i < length; ++i) {
                    char ch = buffer[offset + i];
                    blockBuffer[blockTail] = (byte)(ch >> 8);
                    blockBuffer[blockTail + 1] = (byte)ch;
                    blockTail += 2;
                }
                block.setDirty(blockOffset, blockTail);
                Block block2 = block;
                lock.unlock();
                return block2;
            }
            catch (Throwable throwable) {
                try {
                    lock.unlock();
                    throw throwable;
                }
                catch (InterruptedException e) {
                    throw new IllegalStateException(e);
                }
            }
        }
        finally {
            block.free();
        }
    }

    private int getMiniFragmentOffset(long fragmentAddress) {
        int id = (int)(fragmentAddress & 0x1FFFL);
        return 256 * id;
    }

    public void flush() {
        if (this._lifecycle.isActive()) {
            if (this._blockManager != null) {
                this._blockManager.flush(this);
            }
            long timeout = 100L;
            this.getWriter().waitForComplete(timeout);
        }
    }

    public void fatalCorrupted(String msg) {
        String fullMsg = "caucho.database.corrupt[" + this._path.getFullPath() + "] " + msg;
        HealthSystemFacade.fireFatalEvent(DATABASE_CORRUPT_EVENT, fullMsg);
    }

    public boolean isClosed() {
        return this._lifecycle.isDestroyed();
    }

    public boolean isActive() {
        return this._lifecycle.isActive();
    }

    public void close() {
        if (!this._lifecycle.toDestroy()) {
            return;
        }
        log.finer(this + " closing");
        BlockManager blockManager = this._blockManager;
        if (blockManager != null) {
            blockManager.freeStore(this);
        }
        try {
            this._writer.wake();
            this._writer.waitForComplete(60000L);
        }
        finally {
            this._writer.close();
        }
        int id = this._id;
        this._id = 0;
        this._readWrite.close();
        if (blockManager != null) {
            blockManager.freeStoreId(id);
        }
    }

    public boolean fsync() throws IOException {
        log.finer(this + " fsync");
        this.flush();
        this._writer.wake();
        boolean isValid = this._writer.waitForComplete(60000L);
        this._readWrite.fsync();
        return isValid;
    }

    public void wakeWriter() {
        this._writer.wakeIfPending();
    }

    public byte[] getAllocationTable() {
        byte[] table = new byte[this._allocationTable.length];
        System.arraycopy(this._allocationTable, 0, table, 0, table.length);
        return table;
    }

    public static long readLong(byte[] buffer, int offset) {
        return (((long)buffer[offset + 0] & 0xFFL) << 56) + (((long)buffer[offset + 1] & 0xFFL) << 48) + (((long)buffer[offset + 2] & 0xFFL) << 40) + (((long)buffer[offset + 3] & 0xFFL) << 32) + (((long)buffer[offset + 4] & 0xFFL) << 24) + (((long)buffer[offset + 5] & 0xFFL) << 16) + (((long)buffer[offset + 6] & 0xFFL) << 8) + ((long)buffer[offset + 7] & 0xFFL);
    }

    public static void writeLong(byte[] buffer, int offset, long v) {
        buffer[offset + 0] = (byte)(v >> 56);
        buffer[offset + 1] = (byte)(v >> 48);
        buffer[offset + 2] = (byte)(v >> 40);
        buffer[offset + 3] = (byte)(v >> 32);
        buffer[offset + 4] = (byte)(v >> 24);
        buffer[offset + 5] = (byte)(v >> 16);
        buffer[offset + 6] = (byte)(v >> 8);
        buffer[offset + 7] = (byte)v;
    }

    public static String codeToName(int code) {
        switch (code) {
            case 0: {
                return "free";
            }
            case 1: {
                return "row";
            }
            case 2: {
                return "used";
            }
            case 5: {
                return "mini-fragment";
            }
            case 4: {
                return "index";
            }
        }
        return String.valueOf(code);
    }

    public String toString() {
        return this.getClass().getSimpleName() + "[" + this._id + "," + this._path + "]";
    }
}

