/*
 * Decompiled with CFR 0.152.
 */
package bt.data.file;

import bt.BtException;
import bt.data.StorageUnit;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class FileSystemStorageUnit
implements StorageUnit {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileSystemStorageUnit.class);
    private Path parent;
    private Path file;
    private SeekableByteChannel sbc;
    private long capacity;
    private volatile boolean closed;

    FileSystemStorageUnit(Path root, String path, long capacity) {
        this.file = root.resolve(path);
        this.parent = this.file.getParent();
        this.capacity = capacity;
        this.closed = true;
    }

    private boolean init(boolean create) {
        if (this.closed) {
            if (!Files.exists(this.parent, new LinkOption[0])) {
                try {
                    Files.createDirectories(this.parent, new FileAttribute[0]);
                }
                catch (IOException e) {
                    if (create) {
                        throw new BtException("Failed to create file storage -- can't create (some of the) directories", e);
                    }
                    throw new BtException("Failed to create file storage -- unexpected I/O error", e);
                }
            }
            if (!Files.exists(this.file, new LinkOption[0])) {
                if (create) {
                    try {
                        Files.createFile(this.file, new FileAttribute[0]);
                    }
                    catch (IOException e) {
                        throw new BtException("Failed to create file storage -- can't create new file: " + this.file.toAbsolutePath(), e);
                    }
                } else {
                    return false;
                }
            }
            try {
                this.sbc = Files.newByteChannel(this.file, StandardOpenOption.READ, StandardOpenOption.WRITE);
            }
            catch (IOException e) {
                throw new BtException("Unexpected I/O error", e);
            }
            this.closed = false;
        }
        return true;
    }

    @Override
    public synchronized void readBlock(ByteBuffer buffer, long offset) {
        if (this.closed && !this.init(false)) {
            return;
        }
        if (offset < 0L) {
            throw new BtException("Illegal arguments: offset (" + offset + ")");
        }
        if (offset > this.capacity - (long)buffer.remaining()) {
            throw new BtException("Received a request to read past the end of file (offset: " + offset + ", requested block length: " + buffer.remaining() + ", file size: " + this.capacity);
        }
        try {
            this.sbc.position(offset);
            int read = 1;
            while (buffer.hasRemaining() && read > 0) {
                read = this.sbc.read(buffer);
            }
        }
        catch (IOException e) {
            throw new BtException("Failed to read bytes (offset: " + offset + ", requested block length: " + buffer.remaining() + ", file size: " + this.capacity + ")", e);
        }
    }

    @Override
    public synchronized byte[] readBlock(long offset, int length) {
        if (this.closed && !this.init(false)) {
            return new byte[length];
        }
        if (offset < 0L || length < 0) {
            throw new BtException("Illegal arguments: offset (" + offset + "), length (" + length + ")");
        }
        if (offset > this.capacity - (long)length) {
            throw new BtException("Received a request to read past the end of file (offset: " + offset + ", requested block length: " + length + ", file size: " + this.capacity);
        }
        try {
            this.sbc.position(offset);
            ByteBuffer buf = ByteBuffer.allocate(length);
            int read = 1;
            while (buf.hasRemaining() && read > 0) {
                read = this.sbc.read(buf);
            }
            return buf.array();
        }
        catch (IOException e) {
            throw new BtException("Failed to read bytes (offset: " + offset + ", requested block length: " + length + ", file size: " + this.capacity + ")", e);
        }
    }

    @Override
    public synchronized void writeBlock(ByteBuffer buffer, long offset) {
        if (this.closed) {
            this.init(true);
        }
        if (offset < 0L) {
            throw new BtException("Negative offset: " + offset);
        }
        if (offset > this.capacity - (long)buffer.remaining()) {
            throw new BtException("Received a request to write past the end of file (offset: " + offset + ", block length: " + buffer.remaining() + ", file size: " + this.capacity);
        }
        try {
            this.sbc.position(offset);
            int written = 1;
            while (buffer.hasRemaining() && written > 0) {
                written = this.sbc.write(buffer);
            }
        }
        catch (IOException e) {
            throw new BtException("Failed to write bytes (offset: " + offset + ", block length: " + buffer.remaining() + ", file size: " + this.capacity + ")", e);
        }
    }

    @Override
    public synchronized void writeBlock(byte[] block, long offset) {
        if (this.closed) {
            this.init(true);
        }
        if (offset < 0L) {
            throw new BtException("Negative offset: " + offset);
        }
        if (offset > this.capacity - (long)block.length) {
            throw new BtException("Received a request to write past the end of file (offset: " + offset + ", block length: " + block.length + ", file size: " + this.capacity);
        }
        try {
            this.sbc.position(offset);
            ByteBuffer buf = ByteBuffer.wrap(block);
            int written = 1;
            while (buf.hasRemaining() && written > 0) {
                written = this.sbc.write(buf);
            }
        }
        catch (IOException e) {
            throw new BtException("Failed to write bytes (offset: " + offset + ", block length: " + block.length + ", file size: " + this.capacity + ")", e);
        }
    }

    @Override
    public long capacity() {
        return this.capacity;
    }

    @Override
    public long size() {
        try {
            return Files.exists(this.file, new LinkOption[0]) ? Files.size(this.file) : 0L;
        }
        catch (IOException e) {
            throw new BtException("Unexpected I/O error", e);
        }
    }

    public String toString() {
        return "(" + this.capacity + " B) " + this.file;
    }

    @Override
    public void close() throws IOException {
        if (!this.closed) {
            try {
                this.sbc.close();
            }
            catch (IOException e) {
                LOGGER.warn("Failed to close file: " + this.file, (Throwable)e);
            }
            finally {
                this.closed = true;
            }
        }
    }
}

