/*
 * Decompiled with CFR 0.152.
 */
package bt.peer;

import bt.event.EventSink;
import bt.metainfo.Torrent;
import bt.metainfo.TorrentId;
import bt.net.InetPeer;
import bt.net.Peer;
import bt.peer.IPeerCache;
import bt.peer.IPeerRegistry;
import bt.peer.PeerSource;
import bt.peer.PeerSourceFactory;
import bt.peer.TrackerPeerSourceFactory;
import bt.runtime.Config;
import bt.service.IRuntimeLifecycleBinder;
import bt.service.IdentityService;
import bt.torrent.TorrentDescriptor;
import bt.torrent.TorrentRegistry;
import bt.tracker.AnnounceKey;
import bt.tracker.ITrackerService;
import com.google.inject.Inject;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PeerRegistry
implements IPeerRegistry {
    private static final Logger LOGGER = LoggerFactory.getLogger(PeerRegistry.class);
    private final Peer localPeer;
    private final IPeerCache cache;
    private TorrentRegistry torrentRegistry;
    private ITrackerService trackerService;
    private EventSink eventSink;
    private TrackerPeerSourceFactory trackerPeerSourceFactory;
    private Set<PeerSourceFactory> extraPeerSourceFactories;
    private ConcurrentMap<TorrentId, Set<AnnounceKey>> extraAnnounceKeys;
    private ReentrantLock extraAnnounceKeysLock;

    @Inject
    public PeerRegistry(IRuntimeLifecycleBinder lifecycleBinder, IdentityService idService, TorrentRegistry torrentRegistry, ITrackerService trackerService, EventSink eventSink, IPeerCache cache, Set<PeerSourceFactory> extraPeerSourceFactories, Config config) {
        this.localPeer = new InetPeer(config.getAcceptorAddress(), config.getAcceptorPort(), idService.getLocalPeerId());
        this.cache = cache;
        this.torrentRegistry = torrentRegistry;
        this.trackerService = trackerService;
        this.eventSink = eventSink;
        this.trackerPeerSourceFactory = new TrackerPeerSourceFactory(trackerService, torrentRegistry, lifecycleBinder, config.getTrackerQueryInterval());
        this.extraPeerSourceFactories = extraPeerSourceFactories;
        this.extraAnnounceKeys = new ConcurrentHashMap<TorrentId, Set<AnnounceKey>>();
        this.extraAnnounceKeysLock = new ReentrantLock();
        this.createExecutor(lifecycleBinder, config.getPeerDiscoveryInterval());
    }

    private void createExecutor(IRuntimeLifecycleBinder lifecycleBinder, Duration peerDiscoveryInterval) {
        ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(r -> new Thread(r, "bt.peer.peer-collector"));
        lifecycleBinder.onStartup("Schedule periodic peer lookup", () -> executor.scheduleAtFixedRate(this::collectAndVisitPeers, 1L, peerDiscoveryInterval.toMillis(), TimeUnit.MILLISECONDS));
        lifecycleBinder.onShutdown("Shutdown peer lookup scheduler", executor::shutdownNow);
    }

    private void collectAndVisitPeers() {
        this.torrentRegistry.getTorrentIds().forEach(torrentId -> {
            Optional<TorrentDescriptor> descriptor = this.torrentRegistry.getDescriptor((TorrentId)torrentId);
            if (descriptor.isPresent() && descriptor.get().isActive()) {
                Optional<Torrent> torrentOptional = this.torrentRegistry.getTorrent((TorrentId)torrentId);
                Optional<AnnounceKey> torrentAnnounceKey = torrentOptional.isPresent() ? torrentOptional.get().getAnnounceKey() : Optional.empty();
                Collection extraTorrentAnnounceKeys = (Collection)this.extraAnnounceKeys.get(torrentId);
                if (extraTorrentAnnounceKeys == null) {
                    this.queryTrackers((TorrentId)torrentId, torrentAnnounceKey, (Collection<AnnounceKey>)Collections.emptyList());
                } else if (torrentOptional.isPresent() && torrentOptional.get().isPrivate()) {
                    if (extraTorrentAnnounceKeys.size() > 0) {
                        LOGGER.warn("Will not query extra trackers for a private torrent, id: {}", torrentId);
                    }
                } else {
                    ArrayList<AnnounceKey> extraTorrentAnnounceKeysCopy;
                    this.extraAnnounceKeysLock.lock();
                    try {
                        extraTorrentAnnounceKeysCopy = new ArrayList<AnnounceKey>(extraTorrentAnnounceKeys);
                    }
                    finally {
                        this.extraAnnounceKeysLock.unlock();
                    }
                    this.queryTrackers((TorrentId)torrentId, torrentAnnounceKey, (Collection<AnnounceKey>)extraTorrentAnnounceKeysCopy);
                }
                if (!(torrentOptional.isPresent() && torrentOptional.get().isPrivate() || this.extraPeerSourceFactories.isEmpty())) {
                    this.extraPeerSourceFactories.forEach(factory -> this.queryPeerSource((TorrentId)torrentId, factory.getPeerSource((TorrentId)torrentId)));
                }
            }
        });
    }

    private void queryTrackers(TorrentId torrentId, Optional<AnnounceKey> torrentAnnounceKey, Collection<AnnounceKey> extraAnnounceKeys) {
        torrentAnnounceKey.ifPresent(announceKey -> {
            try {
                this.queryTracker(torrentId, (AnnounceKey)announceKey);
            }
            catch (Exception e) {
                LOGGER.error("Error when querying tracker (torrent's announce key): " + announceKey, (Throwable)e);
            }
        });
        extraAnnounceKeys.forEach(announceKey -> {
            try {
                this.queryTracker(torrentId, (AnnounceKey)announceKey);
            }
            catch (Exception e) {
                LOGGER.error("Error when querying tracker (extra announce key): " + announceKey, (Throwable)e);
            }
        });
    }

    private void queryTracker(TorrentId torrentId, AnnounceKey announceKey) {
        if (this.mightCreateTracker(announceKey)) {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("Querying tracker peer source (announce key: {}) for torrent id: {}", (Object)announceKey, (Object)torrentId);
            }
            this.queryPeerSource(torrentId, this.trackerPeerSourceFactory.getPeerSource(torrentId, announceKey));
        }
    }

    private boolean mightCreateTracker(AnnounceKey announceKey) {
        if (announceKey.isMultiKey()) {
            for (List<String> tier : announceKey.getTrackerUrls()) {
                for (String trackerUrl : tier) {
                    if (this.trackerService.isSupportedProtocol(trackerUrl)) continue;
                    return false;
                }
            }
            return true;
        }
        return this.trackerService.isSupportedProtocol(announceKey.getTrackerUrl());
    }

    private void queryPeerSource(TorrentId torrentId, PeerSource peerSource) {
        try {
            if (peerSource.update()) {
                Collection<Peer> discoveredPeers = peerSource.getPeers();
                Iterator<Peer> iter = discoveredPeers.iterator();
                while (iter.hasNext()) {
                    Peer peer = iter.next();
                    this.addPeer(torrentId, peer);
                    iter.remove();
                }
            }
        }
        catch (Exception e) {
            LOGGER.error("Error when querying peer source: " + peerSource, (Throwable)e);
        }
    }

    @Override
    public void addPeer(TorrentId torrentId, Peer peer) {
        if (this.isLocal(peer)) {
            return;
        }
        this.cache.store(peer);
        this.eventSink.firePeerDiscovered(torrentId, peer);
    }

    @Override
    public void addPeerSource(TorrentId torrentId, AnnounceKey announceKey) {
        this.extraAnnounceKeysLock.lock();
        try {
            this.getOrCreateExtraAnnounceKeys(torrentId).add(announceKey);
        }
        finally {
            this.extraAnnounceKeysLock.unlock();
        }
    }

    private Set<AnnounceKey> getOrCreateExtraAnnounceKeys(TorrentId torrentId) {
        Set existing;
        Set announceKeys = (ConcurrentHashMap.KeySetView)this.extraAnnounceKeys.get(torrentId);
        if (announceKeys == null && (existing = (Set)this.extraAnnounceKeys.putIfAbsent(torrentId, announceKeys = ConcurrentHashMap.newKeySet())) != null) {
            announceKeys = existing;
        }
        return announceKeys;
    }

    private boolean isLocal(Peer peer) {
        return peer.getInetAddress().isAnyLocalAddress() && this.localPeer.getPort() == peer.getPort();
    }

    @Override
    public Peer getLocalPeer() {
        return this.localPeer;
    }
}

