/*
 * Decompiled with CFR 0.152.
 */
package bt.runtime;

import bt.BtException;
import bt.event.EventSource;
import bt.module.ClientExecutor;
import bt.runtime.BtClient;
import bt.runtime.BtRuntimeBuilder;
import bt.runtime.Config;
import bt.service.IRuntimeLifecycleBinder;
import bt.service.LifecycleBinding;
import com.google.inject.Injector;
import com.google.inject.Key;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BtRuntime {
    private static final Logger LOGGER = LoggerFactory.getLogger(BtRuntime.class);
    private Injector injector;
    private Config config;
    private Set<BtClient> knownClients;
    private ExecutorService clientExecutor;
    private AtomicBoolean started;
    private final Object lock;
    private boolean manualShutdownOnly;

    public static BtRuntimeBuilder builder() {
        return new BtRuntimeBuilder();
    }

    public static BtRuntimeBuilder builder(Config config) {
        return new BtRuntimeBuilder(config);
    }

    public static BtRuntime defaultRuntime() {
        return BtRuntime.builder().build();
    }

    BtRuntime(Injector injector, Config config) {
        Runtime.getRuntime().addShutdownHook(new Thread("bt.runtime.shutdown-manager"){

            @Override
            public void run() {
                BtRuntime.this.shutdown();
            }
        });
        this.injector = injector;
        this.config = config;
        this.knownClients = ConcurrentHashMap.newKeySet();
        this.clientExecutor = (ExecutorService)injector.getBinding(Key.get(ExecutorService.class, ClientExecutor.class)).getProvider().get();
        this.started = new AtomicBoolean(false);
        this.lock = new Object();
    }

    void disableAutomaticShutdown() {
        this.manualShutdownOnly = true;
    }

    public Injector getInjector() {
        return this.injector;
    }

    public Config getConfig() {
        return this.config;
    }

    public <T> T service(Class<T> serviceType) {
        return (T)this.injector.getInstance(serviceType);
    }

    public boolean isRunning() {
        return this.started.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void startup() {
        if (this.started.compareAndSet(false, true)) {
            Object object = this.lock;
            synchronized (object) {
                this.runHooks(IRuntimeLifecycleBinder.LifecycleEvent.STARTUP, e -> LOGGER.error("Error on runtime startup", e));
            }
        }
    }

    public void attachClient(BtClient client) {
        this.knownClients.add(client);
    }

    public void detachClient(BtClient client) {
        if (this.knownClients.remove(client)) {
            if (!this.manualShutdownOnly && this.knownClients.isEmpty()) {
                this.shutdown();
            }
        } else {
            throw new IllegalArgumentException("Unknown client: " + client);
        }
    }

    public Collection<BtClient> getClients() {
        return Collections.unmodifiableCollection(this.knownClients);
    }

    public EventSource getEventSource() {
        return this.service(EventSource.class);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() {
        if (this.started.compareAndSet(true, false)) {
            Object object = this.lock;
            synchronized (object) {
                this.knownClients.forEach(client -> {
                    try {
                        client.stop();
                    }
                    catch (Throwable e) {
                        LOGGER.error("Error when stopping client", e);
                    }
                });
                this.runHooks(IRuntimeLifecycleBinder.LifecycleEvent.SHUTDOWN, this::onShutdownHookError);
                this.clientExecutor.shutdownNow();
            }
        }
    }

    private void runHooks(IRuntimeLifecycleBinder.LifecycleEvent event, Consumer<Throwable> errorConsumer) {
        ExecutorService executor = this.createLifecycleExecutor(event);
        HashMap<LifecycleBinding, CompletableFuture> futures = new HashMap<LifecycleBinding, CompletableFuture>();
        ArrayList syncBindings = new ArrayList();
        this.service(IRuntimeLifecycleBinder.class).visitBindings(event, binding -> {
            if (binding.isAsync()) {
                futures.put((LifecycleBinding)binding, CompletableFuture.runAsync(this.toRunnable(event, (LifecycleBinding)binding), executor));
            } else {
                syncBindings.add(binding);
            }
        });
        syncBindings.forEach(binding -> {
            String errorMessage = this.createErrorMessage(event, (LifecycleBinding)binding);
            try {
                this.toRunnable(event, (LifecycleBinding)binding).run();
            }
            catch (Throwable e) {
                errorConsumer.accept(new BtException(errorMessage, e));
            }
        });
        if (event == IRuntimeLifecycleBinder.LifecycleEvent.SHUTDOWN) {
            futures.forEach((binding, future) -> {
                String errorMessage = this.createErrorMessage(event, (LifecycleBinding)binding);
                try {
                    future.get(this.config.getShutdownHookTimeout().toMillis(), TimeUnit.MILLISECONDS);
                }
                catch (InterruptedException | ExecutionException | TimeoutException e) {
                    errorConsumer.accept(new BtException(errorMessage, e));
                }
            });
        }
        this.shutdownGracefully(executor);
    }

    private String createErrorMessage(IRuntimeLifecycleBinder.LifecycleEvent event, LifecycleBinding binding) {
        Optional<String> descriptionOptional = binding.getDescription();
        String errorMessage = "Failed to execute " + event.name().toLowerCase() + " hook: ";
        errorMessage = errorMessage + ": " + (descriptionOptional.isPresent() ? descriptionOptional.get() : binding.getRunnable().toString());
        return errorMessage;
    }

    private ExecutorService createLifecycleExecutor(IRuntimeLifecycleBinder.LifecycleEvent event) {
        AtomicInteger threadCount = new AtomicInteger();
        return Executors.newCachedThreadPool(r -> {
            Thread t = new Thread(r, "bt.runtime." + event.name().toLowerCase() + "-worker-" + threadCount.incrementAndGet());
            t.setDaemon(true);
            return t;
        });
    }

    private void shutdownGracefully(ExecutorService executor) {
        executor.shutdown();
        try {
            long timeout = this.config.getShutdownHookTimeout().toMillis();
            boolean terminated = executor.awaitTermination(timeout, TimeUnit.MILLISECONDS);
            if (!terminated) {
                LOGGER.warn("Failed to shutdown executor in {} millis", (Object)timeout);
            }
        }
        catch (InterruptedException e) {
            LOGGER.warn("Interrupted while waiting for shutdown", (Throwable)e);
            executor.shutdownNow();
        }
    }

    private Runnable toRunnable(IRuntimeLifecycleBinder.LifecycleEvent event, LifecycleBinding binding) {
        return () -> {
            Runnable r = binding.getRunnable();
            Optional<String> descriptionOptional = binding.getDescription();
            String description = descriptionOptional.isPresent() ? descriptionOptional.get() : r.toString();
            LOGGER.debug("Running " + event.name().toLowerCase() + " hook: " + description);
            r.run();
        };
    }

    private void onShutdownHookError(Throwable e) {
        e.printStackTrace(System.err);
        System.err.flush();
    }
}

