/*
 * Decompiled with CFR 0.152.
 */
package bt.torrent.messaging;

import bt.data.Bitfield;
import bt.event.EventSource;
import bt.event.PeerConnectedEvent;
import bt.event.PeerDisconnectedEvent;
import bt.event.PeerDiscoveredEvent;
import bt.metainfo.TorrentId;
import bt.net.IConnectionSource;
import bt.net.IMessageDispatcher;
import bt.net.Peer;
import bt.protocol.Have;
import bt.protocol.Interested;
import bt.protocol.Message;
import bt.protocol.NotInterested;
import bt.runtime.Config;
import bt.torrent.BitfieldBasedStatistics;
import bt.torrent.messaging.Assignment;
import bt.torrent.messaging.Assignments;
import bt.torrent.messaging.ConnectionState;
import bt.torrent.messaging.IPeerWorkerFactory;
import bt.torrent.messaging.PeerWorker;
import java.time.Duration;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TorrentWorker {
    private static final Logger LOGGER = LoggerFactory.getLogger(TorrentWorker.class);
    private static final Duration UPDATE_ASSIGNMENTS_OPTIONAL_INTERVAL = Duration.ofSeconds(1L);
    private static final Duration UPDATE_ASSIGNMENTS_MANDATORY_INTERVAL = Duration.ofSeconds(5L);
    private TorrentId torrentId;
    private IMessageDispatcher dispatcher;
    private Config config;
    private final IConnectionSource connectionSource;
    private IPeerWorkerFactory peerWorkerFactory;
    private ConcurrentMap<Peer, PieceAnnouncingPeerWorker> peerMap;
    private final int MAX_CONCURRENT_ACTIVE_CONNECTIONS;
    private final int MAX_TOTAL_CONNECTIONS;
    private Map<Peer, Long> timeoutedPeers;
    private Queue<Peer> disconnectedPeers;
    private Map<Peer, Message> interestUpdates;
    private long lastUpdatedAssignments;
    private Supplier<Bitfield> bitfieldSupplier;
    private Supplier<Assignments> assignmentsSupplier;
    private Supplier<BitfieldBasedStatistics> statisticsSupplier;

    public TorrentWorker(TorrentId torrentId, IMessageDispatcher dispatcher, IConnectionSource connectionSource, IPeerWorkerFactory peerWorkerFactory, Supplier<Bitfield> bitfieldSupplier, Supplier<Assignments> assignmentsSupplier, Supplier<BitfieldBasedStatistics> statisticsSupplier, EventSource eventSource, Config config) {
        this.torrentId = torrentId;
        this.dispatcher = dispatcher;
        this.config = config;
        this.connectionSource = connectionSource;
        this.peerWorkerFactory = peerWorkerFactory;
        this.peerMap = new ConcurrentHashMap<Peer, PieceAnnouncingPeerWorker>();
        this.MAX_CONCURRENT_ACTIVE_CONNECTIONS = config.getMaxConcurrentlyActivePeerConnectionsPerTorrent();
        this.MAX_TOTAL_CONNECTIONS = config.getMaxPeerConnectionsPerTorrent();
        this.timeoutedPeers = new ConcurrentHashMap<Peer, Long>();
        this.disconnectedPeers = new LinkedBlockingQueue<Peer>();
        this.interestUpdates = new ConcurrentHashMap<Peer, Message>();
        this.bitfieldSupplier = bitfieldSupplier;
        this.assignmentsSupplier = assignmentsSupplier;
        this.statisticsSupplier = statisticsSupplier;
        eventSource.onPeerDiscovered((PeerDiscoveredEvent e) -> {
            if (torrentId.equals(e.getTorrentId())) {
                this.onPeerDiscovered(e.getPeer());
            }
        });
        eventSource.onPeerConnected((PeerConnectedEvent e) -> {
            if (torrentId.equals(e.getTorrentId())) {
                this.onPeerConnected(e.getPeer());
            }
        });
        eventSource.onPeerDisconnected((PeerDisconnectedEvent e) -> {
            if (torrentId.equals(e.getTorrentId())) {
                this.onPeerDisconnected(e.getPeer());
            }
        });
    }

    private Bitfield getBitfield() {
        return this.bitfieldSupplier.get();
    }

    private Assignments getAssignments() {
        return this.assignmentsSupplier.get();
    }

    private BitfieldBasedStatistics getStatistics() {
        return this.statisticsSupplier.get();
    }

    private void addPeer(Peer peer) {
        PieceAnnouncingPeerWorker worker = this.createPeerWorker(peer);
        PieceAnnouncingPeerWorker existing = this.peerMap.putIfAbsent(peer, worker);
        if (existing == null) {
            this.dispatcher.addMessageConsumer(peer, message -> this.consume(peer, (Message)message));
            this.dispatcher.addMessageSupplier(peer, () -> this.produce(peer));
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Added connection for peer: " + peer);
            }
        }
    }

    private void consume(Peer peer, Message message) {
        this.getWorker(peer).ifPresent(worker -> worker.accept(message));
    }

    private Message produce(Peer peer) {
        Message message = null;
        Optional<PieceAnnouncingPeerWorker> workerOptional = this.getWorker(peer);
        if (workerOptional.isPresent()) {
            PieceAnnouncingPeerWorker worker = workerOptional.get();
            Bitfield bitfield = this.getBitfield();
            Assignments assignments = this.getAssignments();
            if (bitfield != null && assignments != null && (bitfield.getPiecesRemaining() > 0 || assignments.count() > 0)) {
                Message interestUpdate;
                this.inspectAssignment(peer, worker, assignments);
                if (this.shouldUpdateAssignments(assignments)) {
                    this.processDisconnectedPeers(assignments, this.getStatistics());
                    this.processTimeoutedPeers();
                    this.updateAssignments(assignments);
                }
                message = (interestUpdate = this.interestUpdates.remove(peer)) == null ? worker.get() : interestUpdate;
            } else {
                message = worker.get();
            }
        }
        return message;
    }

    private Optional<PieceAnnouncingPeerWorker> getWorker(Peer peer) {
        return Optional.ofNullable(this.peerMap.get(peer));
    }

    private void inspectAssignment(Peer peer, PeerWorker peerWorker, Assignments assignments) {
        Optional<Assignment> newAssignment;
        boolean shouldAssign;
        Assignment assignment;
        ConnectionState connectionState;
        block12: {
            block11: {
                connectionState = peerWorker.getConnectionState();
                assignment = assignments.get(peer);
                if (assignment == null) break block11;
                switch (assignment.getStatus()) {
                    case ACTIVE: {
                        shouldAssign = false;
                        break block12;
                    }
                    case DONE: {
                        assignments.remove(assignment);
                        shouldAssign = true;
                        break block12;
                    }
                    case TIMEOUT: {
                        this.timeoutedPeers.put(peer, System.currentTimeMillis());
                        assignments.remove(assignment);
                        shouldAssign = false;
                        if (LOGGER.isTraceEnabled()) {
                            LOGGER.trace("Peer assignment removed due to TIMEOUT: {}", (Object)assignment);
                        }
                        break block12;
                    }
                    default: {
                        throw new IllegalStateException("Unexpected status: " + assignment.getStatus().name());
                    }
                }
            }
            shouldAssign = true;
        }
        if (connectionState.isPeerChoking()) {
            if (assignment != null) {
                assignments.remove(assignment);
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace("Peer assignment removed due to CHOKING: {}", (Object)assignment);
                }
            }
        } else if (shouldAssign && this.mightCreateMoreAssignments(assignments) && (newAssignment = assignments.assign(peer)).isPresent()) {
            newAssignment.get().start(connectionState);
        }
    }

    private boolean shouldUpdateAssignments(Assignments assignments) {
        return this.timeSinceLastUpdated() > UPDATE_ASSIGNMENTS_OPTIONAL_INTERVAL.toMillis() && this.mightUseMoreAssignees(assignments) || this.timeSinceLastUpdated() > UPDATE_ASSIGNMENTS_MANDATORY_INTERVAL.toMillis();
    }

    private boolean mightUseMoreAssignees(Assignments assignments) {
        return assignments.workersCount() < this.MAX_CONCURRENT_ACTIVE_CONNECTIONS;
    }

    private boolean mightCreateMoreAssignments(Assignments assignments) {
        return assignments.count() < this.MAX_CONCURRENT_ACTIVE_CONNECTIONS;
    }

    private long timeSinceLastUpdated() {
        return System.currentTimeMillis() - this.lastUpdatedAssignments;
    }

    private void processDisconnectedPeers(Assignments assignments, BitfieldBasedStatistics statistics) {
        Peer disconnectedPeer;
        while ((disconnectedPeer = this.disconnectedPeers.poll()) != null) {
            Assignment assignment;
            if (assignments != null && (assignment = assignments.get(disconnectedPeer)) != null) {
                assignments.remove(assignment);
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace("Peer assignment removed due to DISCONNECT: peer {}, assignment {}", (Object)disconnectedPeer, (Object)assignment);
                }
            }
            this.timeoutedPeers.remove(disconnectedPeer);
            if (statistics == null) continue;
            statistics.removeBitfield(disconnectedPeer);
        }
    }

    private void processTimeoutedPeers() {
        Iterator<Map.Entry<Peer, Long>> timeoutedPeersIter = this.timeoutedPeers.entrySet().iterator();
        while (timeoutedPeersIter.hasNext()) {
            Map.Entry<Peer, Long> entry = timeoutedPeersIter.next();
            if (System.currentTimeMillis() - entry.getValue() < this.config.getTimeoutedAssignmentPeerBanDuration().toMillis()) continue;
            timeoutedPeersIter.remove();
        }
    }

    private void updateAssignments(Assignments assignments) {
        this.interestUpdates.clear();
        HashSet<Peer> ready = new HashSet<Peer>();
        HashSet<Peer> choking = new HashSet<Peer>();
        this.peerMap.forEach((peer, worker) -> {
            boolean timeouted = this.timeoutedPeers.containsKey(peer);
            boolean disconnected = this.disconnectedPeers.contains(peer);
            if (!timeouted && !disconnected) {
                if (worker.getConnectionState().isPeerChoking()) {
                    choking.add((Peer)peer);
                } else {
                    ready.add((Peer)peer);
                }
            }
        });
        Set<Peer> interesting = assignments.update(ready, choking);
        ready.stream().filter(peer -> !interesting.contains(peer)).forEach(peer -> this.getWorker((Peer)peer).ifPresent(worker -> {
            ConnectionState connectionState = worker.getConnectionState();
            if (connectionState.isInterested()) {
                this.interestUpdates.put((Peer)peer, NotInterested.instance());
                connectionState.setInterested(false);
            }
        }));
        choking.forEach(peer -> this.getWorker((Peer)peer).ifPresent(worker -> {
            ConnectionState connectionState = worker.getConnectionState();
            if (interesting.contains(peer)) {
                if (!connectionState.isInterested()) {
                    this.interestUpdates.put((Peer)peer, Interested.instance());
                    connectionState.setInterested(true);
                }
            } else if (connectionState.isInterested()) {
                this.interestUpdates.put((Peer)peer, NotInterested.instance());
                connectionState.setInterested(false);
            }
        }));
        this.lastUpdatedAssignments = System.currentTimeMillis();
    }

    private PieceAnnouncingPeerWorker createPeerWorker(Peer peer) {
        return new PieceAnnouncingPeerWorker(this.peerWorkerFactory.createPeerWorker(this.torrentId, peer));
    }

    public void removePeer(Peer peer) {
        PeerWorker removed = (PeerWorker)this.peerMap.remove(peer);
        if (removed != null) {
            this.disconnectedPeers.add(peer);
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Removed connection for peer: " + peer);
            }
        }
    }

    public Set<Peer> getPeers() {
        return this.peerMap.keySet();
    }

    public ConnectionState getConnectionState(Peer peer) {
        PeerWorker worker = (PeerWorker)this.peerMap.get(peer);
        return worker == null ? null : worker.getConnectionState();
    }

    private synchronized void onPeerDiscovered(Peer peer) {
        if (this.mightAddPeer(peer)) {
            this.connectionSource.getConnectionAsync(peer, this.torrentId);
        }
    }

    private synchronized void onPeerConnected(Peer peer) {
        if (this.mightAddPeer(peer)) {
            this.addPeer(peer);
        }
    }

    private boolean mightAddPeer(Peer peer) {
        return this.getPeers().size() < this.MAX_TOTAL_CONNECTIONS && !this.getPeers().contains(peer);
    }

    private synchronized void onPeerDisconnected(Peer peer) {
        this.removePeer(peer);
    }

    private class PieceAnnouncingPeerWorker
    implements PeerWorker {
        private final PeerWorker delegate;
        private final Queue<Have> pieceAnnouncements;

        PieceAnnouncingPeerWorker(PeerWorker delegate) {
            this.delegate = delegate;
            this.pieceAnnouncements = new ConcurrentLinkedQueue<Have>();
        }

        @Override
        public ConnectionState getConnectionState() {
            return this.delegate.getConnectionState();
        }

        @Override
        public void accept(Message message) {
            this.delegate.accept(message);
        }

        @Override
        public Message get() {
            Message message = this.pieceAnnouncements.poll();
            if (message != null) {
                return message;
            }
            message = (Message)this.delegate.get();
            if (message != null && Have.class.equals(message.getClass())) {
                Have have = (Have)message;
                TorrentWorker.this.peerMap.values().forEach(worker -> {
                    if (this != worker) {
                        worker.getPieceAnnouncements().add(have);
                    }
                });
            }
            return message;
        }

        Queue<Have> getPieceAnnouncements() {
            return this.pieceAnnouncements;
        }
    }
}

