/*
 * Decompiled with CFR 0.152.
 */
package bt.tracker.udp;

import bt.BtException;
import bt.protocol.Protocols;
import bt.service.IRuntimeLifecycleBinder;
import bt.tracker.udp.ConnectRequest;
import bt.tracker.udp.ConnectResponseHandler;
import bt.tracker.udp.Session;
import bt.tracker.udp.UdpTrackerMessage;
import bt.tracker.udp.UdpTrackerResponseHandler;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.SocketAddress;
import java.net.SocketException;
import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class UdpMessageWorker {
    private static final Logger LOGGER = LoggerFactory.getLogger(UdpMessageWorker.class);
    private static final int MIN_MESSAGE_LENGTH = 16;
    private static final int MESSAGE_TYPE_OFFSET = 0;
    private static final int MESSAGE_ID_OFFSET = 4;
    private static final int ERROR_MESSAGE_TYPE = 3;
    private static final int DATA_OFFSET = 8;
    private final SocketAddress localAddress;
    private final SocketAddress remoteAddress;
    private volatile boolean shutdown;
    private volatile DatagramSocket socket;
    private final Object lock;
    private final ExecutorService executor;
    private volatile Session session;

    public UdpMessageWorker(SocketAddress localAddress, SocketAddress remoteAddress, IRuntimeLifecycleBinder lifecycleBinder) {
        this.localAddress = localAddress;
        this.remoteAddress = remoteAddress;
        this.lock = new Object();
        this.executor = Executors.newSingleThreadExecutor(r -> new Thread(r, "bt.tracker.udp.message-worker"));
        lifecycleBinder.onShutdown("Shutdown UDP message worker", () -> {
            try {
                this.shutdown();
            }
            finally {
                this.executor.shutdownNow();
            }
        });
    }

    public synchronized <T> T sendMessage(UdpTrackerMessage message, UdpTrackerResponseHandler<T> responseHandler) {
        return this.sendMessage(message, this.getSession(), responseHandler, false);
    }

    private Session getSession() {
        if (this.session == null || this.session.isExpired()) {
            this.session = this.createSession();
        }
        return this.session;
    }

    private Session createSession() {
        return this.sendMessage(new ConnectRequest(), Session.noSession(), ConnectResponseHandler.handler(), false);
    }

    private <T> T sendMessage(UdpTrackerMessage message, Session session, UdpTrackerResponseHandler<T> responseHandler, boolean retry) {
        int timeToWait = retry ? 5 : 10;
        try {
            return (T)CompletableFuture.supplyAsync(() -> this.doSend(message, session, responseHandler), this.executor).get(timeToWait, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            throw new BtException("Unexpectedly interrupted while waiting for response from the tracker", e);
        }
        catch (TimeoutException e) {
            if (retry) {
                throw new BtException("Failed to receive response from the tracker", e);
            }
            return this.sendMessage(message, session, responseHandler, true);
        }
        catch (Throwable e) {
            throw new BtException("Failed to receive response from the tracker", e);
        }
    }

    private <T> T doSend(UdpTrackerMessage message, Session session, UdpTrackerResponseHandler<T> responseHandler) {
        DatagramSocket socket = this.getSocket();
        try {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace("[Session {}] Sending message {} to remote address: {}", new Object[]{session.getId(), message, this.remoteAddress});
            }
            socket.send(this.serialize(message, session));
            DatagramPacket response = new DatagramPacket(new byte[8192], 8192);
            while (true) {
                socket.receive(response);
                if (!this.remoteAddress.equals(response.getSocketAddress())) {
                    if (!LOGGER.isTraceEnabled()) continue;
                    LOGGER.trace("[Session {}] Received unexpected datagram packet from remote address: {}", (Object)session.getId(), (Object)response.getSocketAddress());
                    continue;
                }
                if (response.getLength() >= 16) {
                    byte[] data = response.getData();
                    int messageType = Protocols.readInt(data, 0);
                    if (messageType == 3) {
                        String error = new String(Arrays.copyOfRange(data, 8, response.getLength()), "ASCII");
                        if (LOGGER.isTraceEnabled()) {
                            LOGGER.trace("[Session {}] Received error from remote address: {}; message ID: {}, messageType: {}, error: {}", new Object[]{session.getId(), this.remoteAddress, message.getId(), messageType, error});
                        }
                        return responseHandler.onError(error);
                    }
                    if (messageType != message.getMessageType()) {
                        if (!LOGGER.isTraceEnabled()) continue;
                        LOGGER.trace("[Session {}] Received message with incorrect type from remote address: {}; expected: {}, actual: {}", new Object[]{session.getId(), this.remoteAddress, message.getMessageType(), messageType});
                        continue;
                    }
                    int messageId = Protocols.readInt(data, 4);
                    if (messageId != message.getId()) {
                        if (!LOGGER.isTraceEnabled()) continue;
                        LOGGER.trace("[Session {}] Received message with incorrect message ID from remote address: {}; expected: {}, actual: {}", new Object[]{session.getId(), this.remoteAddress, message.getId(), messageId});
                        continue;
                    }
                    T result = responseHandler.onSuccess(Arrays.copyOfRange(data, 8, response.getLength()));
                    if (LOGGER.isTraceEnabled()) {
                        LOGGER.trace("[Session {}] Received response from remote address: {}; message ID: {}, messageType: {}, result: {}", new Object[]{session.getId(), this.remoteAddress, messageId, messageType, result});
                    }
                    return result;
                }
                if (!LOGGER.isTraceEnabled()) continue;
                LOGGER.trace("[Session {}] Received message with incorrect size from remote address: {}; expected: at least {} bytes, actual: {} bytes", new Object[]{session.getId(), this.remoteAddress, 16, response.getLength()});
            }
        }
        catch (IOException e) {
            throw new BtException("Interaction with the tracker failed {remoteAddress=" + this.remoteAddress + "}", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private DatagramSocket getSocket() {
        Object object;
        if (this.shutdown) {
            throw new IllegalStateException("Worker is shutdown");
        }
        if (this.socket == null || this.socket.isClosed()) {
            object = this.lock;
            synchronized (object) {
                if (this.socket == null || this.socket.isClosed()) {
                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.debug("Creating UDP socket {localAddress=" + this.localAddress + "}");
                    }
                    this.socket = UdpMessageWorker.createSocket(this.localAddress);
                }
            }
        }
        if (!this.socket.isConnected()) {
            object = this.lock;
            synchronized (object) {
                if (!this.socket.isConnected()) {
                    try {
                        if (LOGGER.isDebugEnabled()) {
                            LOGGER.debug("Connecting UDP socket {localAddress=" + this.localAddress + ", remoteAddress=" + this.remoteAddress + "}");
                        }
                        this.socket.connect(this.remoteAddress);
                    }
                    catch (SocketException e) {
                        throw new BtException("Failed to connect to the tracker {remoteAddress=" + this.remoteAddress + "}", e);
                    }
                }
            }
        }
        return this.socket;
    }

    private static DatagramSocket createSocket(SocketAddress address) {
        try {
            return new DatagramSocket(address);
        }
        catch (SocketException e) {
            throw new BtException("Failed to create socket {localAddress=" + address + "}", e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private DatagramPacket serialize(UdpTrackerMessage message, Session session) {
        try (ByteArrayOutputStream out = new ByteArrayOutputStream();){
            out.write(Protocols.getLongBytes(session.getId()));
            message.writeTo(out);
            DatagramPacket datagramPacket = new DatagramPacket(out.toByteArray(), out.size());
            return datagramPacket;
        }
        catch (IOException e) {
            throw new BtException("Failed to serialize message", e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() {
        Object object = this.lock;
        synchronized (object) {
            if (!this.shutdown) {
                this.shutdown = true;
                this.executor.shutdownNow();
                if (this.socket != null) {
                    this.socket.close();
                }
            }
        }
    }
}

