/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.utility.primitive;

import com.gs.collections.api.BooleanIterable;
import com.gs.collections.api.LazyBooleanIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.block.function.primitive.BooleanToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.BooleanPredicate;
import com.gs.collections.impl.factory.primitive.BooleanLists;
import com.gs.collections.impl.lazy.primitive.CollectBooleanToObjectIterable;
import com.gs.collections.impl.lazy.primitive.LazyBooleanIterableAdapter;
import com.gs.collections.impl.lazy.primitive.SelectBooleanIterable;

/**
 * LazyBooleanIterate is a factory class which creates "deferred" boolean iterables around the specified boolean iterables. A "deferred"
 * boolean iterable performs some operation, such as filtering or transforming, when the result boolean iterable is iterated over.  This
 * makes the operation very memory efficient, because you don't have to create intermediate collections during the
 * operation.
 * This file was automatically generated from template file lazyPrimitiveIterate.stg.
 *
 * @since 5.0
 */
public final class LazyBooleanIterate
{
    private static final LazyBooleanIterable EMPTY_ITERABLE = BooleanLists.immutable.of().asLazy();

    private LazyBooleanIterate()
    {
        throw new AssertionError("Suppress default constructor for noninstantiability");
    }

    /**
     * Creates a deferred boolean iterable for the specified boolean iterable.
     */
    public static LazyBooleanIterable adapt(BooleanIterable iterable)
    {
        return new LazyBooleanIterableAdapter(iterable);
    }

    /**
     * Creates a deferred filtering boolean iterable for the specified boolean iterable.
     */
    public static LazyBooleanIterable select(BooleanIterable iterable, BooleanPredicate predicate)
    {
        return new SelectBooleanIterable(iterable, predicate);
    }

    /**
     * Creates a deferred transforming boolean iterable for the specified boolean iterable.
     */
    public static <V> LazyIterable<V> collect(
            BooleanIterable iterable,
            BooleanToObjectFunction<? extends V> function)
    {
        return new CollectBooleanToObjectIterable<V>(iterable, function);
    }

    /**
     * Creates a deferred filtering and transforming boolean iterable for the specified boolean iterable.
     */
    public static <V> LazyIterable<V> collectIf(
            BooleanIterable iterable,
            BooleanPredicate predicate,
            BooleanToObjectFunction<? extends V> function)
    {
        return LazyBooleanIterate.select(iterable, predicate).collect(function);
    }

    public static LazyBooleanIterable empty()
    {
        return EMPTY_ITERABLE;
    }
}
