/*
 * Copyright (C) 2015 Square, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.jakewharton.retrofit2.adapter.reactor;

import java.io.IOException;
import java.lang.annotation.Annotation;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Scheduler;
import retrofit2.CallAdapter;
import retrofit2.Response;
import retrofit2.Retrofit;

/**
 * A {@linkplain CallAdapter.Factory call adapter} which uses Project Reactor for creating streams.
 * <p>
 * Adding this class to {@link Retrofit} allows you to return a {@link Flux} or {@link Mono} from
 * service methods.
 * <pre><code>
 * interface MyService {
 *   &#64;GET("user/me")
 *   Flux&lt;User&gt; getUser()
 * }
 * </code></pre>
 * There are three configurations supported for the {@code Flux} or {@code Mono} type
 * parameter:
 * <ul>
 * <li>Direct body (e.g., {@code Flux<User>}) calls {@code onNext} with the deserialized body for
 * 2XX responses and calls {@code onError} with {@link retrofit2.HttpException} for non-2XX
 * responses and {@link IOException} for network errors.</li>
 * <li>Response wrapped body (e.g., {@code Flux<Response<User>>}) calls {@code onNext} with a
 * {@link Response} object for all HTTP responses and calls {@code onError} with {@link IOException}
 * for network errors</li>
 * <li>Result wrapped body (e.g., {@code Flux<Result<User>>}) calls {@code onNext} with a
 * {@link Result} object for all HTTP responses and errors.</li>
 * </ul>
 */
public final class ReactorCallAdapterFactory extends CallAdapter.Factory {
  /**
   * Returns an instance which creates synchronous observables that do not operate on any scheduler
   * by default.
   */
  public static ReactorCallAdapterFactory create() {
    return new ReactorCallAdapterFactory(null, false);
  }

  /**
   * Returns an instance which creates asynchronous observables. Applying
   * {@link Flux#subscribeOn} has no effect on stream types created by this factory.
   */
  public static ReactorCallAdapterFactory createAsync() {
    return new ReactorCallAdapterFactory(null, true);
  }

  /**
   * Returns an instance which creates synchronous observables that
   * {@linkplain Flux#publishOn(Scheduler) publishes on} {@code scheduler} by default.
   */
  public static ReactorCallAdapterFactory createWithScheduler(Scheduler scheduler) {
    if (scheduler == null) throw new NullPointerException("scheduler == null");
    return new ReactorCallAdapterFactory(scheduler, false);
  }

  private final Scheduler scheduler;
  private final boolean isAsync;

  private ReactorCallAdapterFactory(Scheduler scheduler, boolean isAsync) {
    this.scheduler = scheduler;
    this.isAsync = isAsync;
  }

  @Override
  public CallAdapter<?, ?> get(Type returnType, Annotation[] annotations, Retrofit retrofit) {
    Class<?> rawType = getRawType(returnType);
    boolean isMono = rawType == Mono.class;
    if (rawType != Flux.class && !isMono) {
      return null;
    }

    boolean isResult = false;
    boolean isBody = false;
    Type responseType;
    if (!(returnType instanceof ParameterizedType)) {
      String name = isMono ? "Mono" : "Flux";
      throw new IllegalStateException(name + " return type must be parameterized"
          + " as " + name + "<Foo> or " + name + "<? extends Foo>");
    }

    Type observableType = getParameterUpperBound(0, (ParameterizedType) returnType);
    Class<?> rawObservableType = getRawType(observableType);
    if (rawObservableType == Response.class) {
      if (!(observableType instanceof ParameterizedType)) {
        throw new IllegalStateException("Response must be parameterized"
            + " as Response<Foo> or Response<? extends Foo>");
      }
      responseType = getParameterUpperBound(0, (ParameterizedType) observableType);
    } else if (rawObservableType == Result.class) {
      if (!(observableType instanceof ParameterizedType)) {
        throw new IllegalStateException("Result must be parameterized"
            + " as Result<Foo> or Result<? extends Foo>");
      }
      responseType = getParameterUpperBound(0, (ParameterizedType) observableType);
      isResult = true;
    } else {
      responseType = observableType;
      isBody = true;
    }

    return new ReactorCallAdapter(responseType, scheduler, isAsync, isResult, isBody, isMono);
  }
}
