package com.mockrunner.test.jdbc;

import java.sql.SQLException;
import java.sql.Statement;
import java.util.Arrays;
import java.util.Properties;

import junit.framework.TestCase;

import com.mockrunner.mock.jdbc.MockArray;
import com.mockrunner.mock.jdbc.MockBlob;
import com.mockrunner.mock.jdbc.MockClob;
import com.mockrunner.mock.jdbc.MockConnection;
import com.mockrunner.mock.jdbc.MockDatabaseMetaData;
import com.mockrunner.mock.jdbc.MockPreparedStatement;
import com.mockrunner.mock.jdbc.MockResultSet;
import com.mockrunner.mock.jdbc.MockSQLXML;
import com.mockrunner.mock.jdbc.MockStruct;

public class MockConnectionTest extends TestCase
{
    private MockConnection connection;

    protected void setUp() throws Exception
    {
        super.setUp();
        connection = new MockConnection();
    }
    
    protected void tearDown() throws Exception
    {
        super.tearDown();
        connection = null;
    }

    public void testDatabaseMetaData() throws SQLException
    {
        assertNotNull(connection.getMetaData());
        assertTrue(connection.getMetaData() instanceof MockDatabaseMetaData);
        assertSame(connection, connection.getMetaData().getConnection());
        connection.setMetaData(null);
        assertNull(connection.getMetaData());
        connection.setMetaData(new MockDatabaseMetaData());
        assertNotNull(connection.getMetaData());
        assertTrue(connection.getMetaData() instanceof MockDatabaseMetaData);
        assertSame(connection, connection.getMetaData().getConnection());
    }
    
    public void testPrepareStatementAutoGeneratedKeys() throws SQLException
    {
        MockResultSet resultSet = new MockResultSet("testid");
        try
        {
            connection.prepareStatement("select", 50000);
        } 
        catch(SQLException exc)
        {
            //should throw exception
        }
        MockPreparedStatement statement = (MockPreparedStatement)connection.prepareStatement("select", Statement.RETURN_GENERATED_KEYS);
        connection.getPreparedStatementResultSetHandler().prepareGeneratedKeys("select", resultSet);
        statement.execute();
        assertEquals("testid", ((MockResultSet)statement.getGeneratedKeys()).getId());
        statement = (MockPreparedStatement)connection.prepareStatement("select", Statement.NO_GENERATED_KEYS);
        statement.execute();
        assertTrue(((MockResultSet)statement.getGeneratedKeys()).getId().indexOf("testid") < 0);
    }
    
    public void testClientInfo() throws SQLException
    {
        assertNull(connection.getClientInfo("name"));
        assertTrue(connection.getClientInfo().isEmpty());
        connection.getClientInfo().setProperty("name", "value");
        assertNull(connection.getClientInfo("name"));
        assertTrue(connection.getClientInfo().isEmpty());
        connection.setClientInfo("name", "value");
        assertEquals("value", connection.getClientInfo("name"));
        assertEquals("value", connection.getClientInfo().getProperty("name"));
        connection.setClientInfo(new Properties());
        assertNull(connection.getClientInfo("name"));
        assertTrue(connection.getClientInfo().isEmpty());
        Properties properties = new Properties();
        properties.setProperty("name", "value");
        connection.setClientInfo(properties);
        assertEquals(1, connection.getClientInfo().size());
        assertEquals("value", connection.getClientInfo().getProperty("name"));
        assertEquals("value", connection.getClientInfo("name"));
    }
    
    public void testCreateTypes() throws SQLException
    {
        MockBlob blob = (MockBlob)connection.createBlob();
        assertEquals(0, blob.length());
        MockClob clob = (MockClob)connection.createClob();
        assertEquals(0, clob.length());
        MockArray array = (MockArray)connection.createArrayOf("aName", new Object[] {"1", "2", "3"});
        assertEquals("aName", array.getBaseTypeName());
        assertTrue(Arrays.equals(new Object[] {"1", "2", "3"}, (Object[])array.getArray()));
        MockStruct struct = (MockStruct)connection.createStruct("aName", new Object[] {"1", "2", "3"});
        assertEquals("aName", struct.getSQLTypeName());
        Object[] attributes = struct.getAttributes();
        assertEquals("1", attributes[0]);
        assertEquals("2", attributes[1]);
        assertEquals("3", attributes[2]);
        MockSQLXML sqlXML = (MockSQLXML)connection.createSQLXML();
        assertNull(sqlXML.getContentAsString());
    }
}
