/**
 * Copyright 2009-2019 the original author or authors.
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * <p>
 * http://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.pcauto.lib.mybatis.keygen;

import com.baomidou.mybatisplus.core.metadata.TableInfo;
import org.apache.ibatis.executor.Executor;
import org.apache.ibatis.executor.ExecutorException;
import org.apache.ibatis.executor.keygen.KeyGenerator;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.reflection.MetaObject;
import org.apache.ibatis.session.Configuration;

import java.sql.Connection;
import java.sql.Statement;

/**
 * @author Clinton Begin
 * @author Jeff Butler
 * @author henry
 */
public class GlKeyGenerator implements KeyGenerator {

    private TableInfo tableInfo;
    private IdGeneratorByTable idGenerator= new IdGeneratorByTable();

    private boolean executeBefore;

    public GlKeyGenerator(TableInfo tableInfo, boolean executeBefore) {
        this.tableInfo = tableInfo;
        this.executeBefore = executeBefore;
    }

    @Override
    public void processBefore(Executor executor, MappedStatement ms, Statement stmt, Object parameter) {
        if (executeBefore) {
            processGeneratedKeys(executor, ms, parameter);
        }
    }

    @Override
    public void processAfter(Executor executor, MappedStatement ms, Statement stmt, Object parameter) {
        if (!executeBefore) {
            processGeneratedKeys(executor, ms, parameter);
        }
    }

    private void processGeneratedKeys(Executor executor, MappedStatement ms, Object parameter) {
        try {
            executor.getTransaction().getConnection();
            final Configuration configuration = ms.getConfiguration();

            String property = "id";
            String keySequence = tableInfo.getKeySequence().value();
            if (parameter != null) {
                final MetaObject metaParam = configuration.newMetaObject(parameter);
                Connection conn=executor.getTransaction().getConnection();
                setValue(metaParam, property, idGenerator.generate(conn, keySequence, property));
            }
        } catch (ExecutorException e) {
            throw e;
        } catch (Exception e) {
            throw new ExecutorException("Error selecting key or setting result to parameter object. Cause: " + e, e);
        }
    }

    private void setValue(MetaObject metaParam, String property, Object value) {
        if (metaParam.hasSetter(property)) {
            metaParam.setValue(property, value);
        } else {
            throw new ExecutorException("No setter found for the keyProperty '" + property + "' in " + metaParam.getOriginalObject().getClass().getName() + ".");
        }
    }
}
