package com.pcauto.lib.codegen.genner;

import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import freemarker.template.TemplateExceptionHandler;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class JpaCodeGenner {
    public static final String TEST = "Test";
    public static final String ENTITY = "Entity";

    private Configuration cfg = new Configuration(Configuration.getVersion());

    /**
     * 初始化freemarker
     *
     * @param tmplPackage
     */
    private void initConfig(String tmplPackage) {
        cfg.setClassForTemplateLoading(this.getClass(), tmplPackage);
        cfg.setDefaultEncoding("UTF-8");
        cfg.setTemplateExceptionHandler(TemplateExceptionHandler.RETHROW_HANDLER);
        cfg.setLogTemplateExceptions(true);
        //cfg.setWrapUncheckedExceptions(true);
    }

    /**
     * @param tmplPackage 模板所在的包路径 如:"/tmpl/jpa"
     * @param dataModel   包名及Entity名
     * @param baseOutput  maven项目的根目录,即pom.xml所在的目录
     */
    public void gen(String tmplPackage, DataModel dataModel, File baseOutput) {
        // 配置freemarker
        initConfig(tmplPackage);

        // 收集模板
        List<TmplFileInfo> tmplFileList = collectTemplates(tmplPackage, tmplPackage);

        // 筛选模板: 从input获取数据，用于部分生成
        String input = getInput(tmplFileList);

        tmplFileList
                .stream()
                .filter(tmplFileInfo -> input.contains(tmplFileInfo.getShortCode()))
                .forEach(tmplFileInfo -> internalGen(tmplFileInfo, dataModel, baseOutput));

    }

    /**
     * 内部生成逻辑
     *
     * @param tmplFileInfo 模板
     * @param data         数据
     * @param baseOutput   输出
     */
    private void internalGen(TmplFileInfo tmplFileInfo, DataModel data, File baseOutput) {
        //根据模板路径和文件名，转换成输出的最终文件名
        //对 Test  Tmpl 做了特殊处理，并且去掉模板后缀
        String newPath = data.getBasePackage()
                .replaceAll("\\.", "/") + "/" + tmplFileInfo.getTemplate()//包名换路径
                //.replaceAll("\\.ftl", "")// 去掉ftl后缀
                .replaceAll("(.*)/([^\\.]*)\\.([^\\.]*).ftl$", "$1/" + data.getEntityName() + "$2.$3")//替换Tmpl文件名前缀
                .replaceAll(ENTITY, "")//Entity 特殊处理
                ;

        try {
            Template temp = cfg.getTemplate(tmplFileInfo.getTemplate());
            //根据模板文件名里的Test关键字，决定生成的最终目录
            // 根据baseOutpu拼装 src/main/java 和 src/test/java
            String javaOut = baseOutput.getPath() + "/src/main/java";
            String testOut = baseOutput.getPath() + "/src/test/java";
            String prefixPath = (newPath.endsWith(TEST + ".java") ? testOut : javaOut);
            File outFile = new File(prefixPath + "/" + newPath);
            if (!outFile.getParentFile().isDirectory()) {
                outFile.getParentFile().mkdirs();
            }
            FileWriter fileWriter = new FileWriter(outFile);
            temp.process(data, fileWriter);
            // Note: Depending on what `out` is, you may need to call `out.close()`.
            // This is usually the case for file output, but not for servlet output.
        } catch (IOException e) {
            e.printStackTrace();
        } catch (TemplateException e) {
            e.printStackTrace();
        }
    }

    private String getInput(List<TmplFileInfo> tmplList) {
        List<String> templates = tmplList.stream()
                .map(TmplFileInfo::getTemplate)
                .filter(s -> !s.contains(TEST))
                .map(s -> s.substring(s.lastIndexOf("/") + 1, s.indexOf(".")))
                .collect(Collectors.toList());
        System.out.println("输入需要生成的类(首字母):如erscvm");
        System.out.println(templates);

        Scanner scan = new Scanner(System.in);
        String read = scan.nextLine();

        //String allLevel = "erscvm";
        return read;//? read : allLevel;
    }

    /**
     * 收集template
     *
     * @param tmplPackage
     * @param rootPackage
     * @return
     */
    private List<TmplFileInfo> collectTemplates(String tmplPackage, String rootPackage) {
        List<TmplFileInfo> collected = new ArrayList<>();
        try (Stream<Path> paths = Files.list(Paths.get(JpaCodeGenner.class.getResource(tmplPackage).toURI()))) {
            paths.forEach(path -> {
                File file = path.toFile();
                if (file.isDirectory()) {
                    collected.addAll(collectTemplates(tmplPackage + "/" + file.getName(), rootPackage));
                } else if (file.getName().endsWith(".ftl")) {
                    collected.add(new TmplFileInfo(path, rootPackage));
                }
            });
        } catch (IOException e) {
            e.printStackTrace();
        } catch (URISyntaxException e) {
            e.printStackTrace();
        }

        return collected;
    }
}
