package com.pcauto.lib.discuss.service;

import com.pcauto.lib.discuss.api.DiscussApi;
import com.pcauto.lib.discuss.api.DiscussResponse;
import com.pcauto.lib.discuss.config.DiscussProperties;
import com.pcauto.lib.discuss.error.DiscussException;
import com.pcauto.lib.discuss.pojo.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;
import retrofit2.Call;
import retrofit2.Response;

import java.io.IOException;
import java.util.Date;
import java.util.List;

/**
 * 评论服务
 *
 * @author sam
 * @since 2019/8/16
 */
@Service
public class DiscussService {

    public static final Logger LOGGER = LoggerFactory.getLogger(DiscussService.class);

    @Autowired
    private DiscussProperties discussProperties;

    @Autowired
    private DiscussApi discussApi;

    /**
     * 创建or更新用户信息
     *
     * @param discussUser
     * @throws DiscussException
     */
    public void saveOrUpdate(DiscussUser discussUser) throws DiscussException {
        String appId = discussProperties.getAppid();
        if (StringUtils.isEmpty(appId)) {
            throw new DiscussException("appId没有正确配置");
        }
        if (StringUtils.isEmpty(discussUser.getUserId())) {
            throw new DiscussException("userId 不能为空");
        }
        if (StringUtils.isEmpty(discussUser.getUserName())) {
            throw new DiscussException("userName 不能为空");
        }
        Call<DiscussResponse> call = discussApi.saveOrUpdate(appId, discussUser);
        checkResponse(call);
    }


    /**
     * 发表评论
     * <p>
     * 异常错误码:
     * 40000: 参数异常
     * 40001: 回复的评论不存在
     * 40002: 不能回复自己
     * 50000: 评论服务访问异常
     *
     * @param commentDto 请求dto
     */
    public void submitComment(DiscussComment commentDto) throws DiscussException {
        String appId = discussProperties.getAppid();
        if (StringUtils.isEmpty(appId)) {
            throw new DiscussException("appId没有正确配置");
        }
        if (StringUtils.isEmpty(commentDto.getUserId())) {
            throw new DiscussException("userId 不能为空");
        }
        if (StringUtils.isEmpty(commentDto.getKey())) {
            throw new DiscussException("评论归属key 不能为空");
        }
        if (StringUtils.isEmpty(commentDto.getContent())) {
            throw new DiscussException("评论内容 不能为空");
        }

        Call<DiscussResponse> call = discussApi.submitComment(appId, commentDto);
        checkResponse(call);
    }

    /**
     * 获取某个key下的评论列表
     * <p>
     * 异常错误码:
     * 40000: 参数异常
     * 50000: 评论服务访问异常
     *
     * @param key      评论归属key [必须]
     * @param userId   当前登录的用户id，用于识别是否有点赞 [非必须]
     * @param page     第几页
     * @param pageSize 每页大小
     * @return
     */
    public DiscussPageResult<DiscussCommentResult> pageComment(String key, String userId, Integer page, Integer pageSize) throws DiscussException {
        return pageComment(key, userId, "asc", page, pageSize);
    }

    /**
     * 获取某个key下的评论列表
     * <p>
     * 异常错误码:
     * 40000: 参数异常
     *
     * @param key      评论归属key [必须]
     * @param userId   当前登录的用户id，用于识别是否有点赞 [非必须]
     * @param sort     排序 desc,asc 默认为:asc [非必须]
     * @param page     第几页
     * @param pageSize 每页大小
     * @return
     */
    public DiscussPageResult<DiscussCommentResult> pageComment(String key, String userId, String sort, Integer page, Integer pageSize) throws DiscussException {
        return pageComment(key, userId, null, sort, page, pageSize);
    }

    /**
     * 获取某个key下的评论列表
     * <p>
     * 异常错误码:
     * 40000: 参数异常
     *
     * @param key      评论归属key [必须]
     * @param userId   当前登录的用户id，用于识别是否有点赞 [非必须]
     * @param sort     排序 desc,asc 默认为:asc [非必须]
     * @param topSort  置顶排序 desc,asc 默认为:asc [非必须]
     * @param page     第几页
     * @param pageSize 每页大小
     * @return
     */
    public DiscussPageResult<DiscussCommentResult> pageComment(String key, String userId, String topSort, String sort, Integer page, Integer pageSize) throws DiscussException {
        String appId = discussProperties.getAppid();
        if (StringUtils.isEmpty(appId)) {
            throw new DiscussException("appId没有正确配置");
        }
        if (StringUtils.isEmpty(key)) {
            throw new DiscussException("key不能为空");
        }
        if (StringUtils.isEmpty(sort)) {
            sort = "asc";
        } else {
            if (!"asc".equals(sort) && !"desc".equals(sort)) {
                throw new DiscussException("sort值有误,请传入asc或desc");
            }
        }
        if (StringUtils.isEmpty(topSort)) {
            topSort = "asc";
        } else {
            if (!"asc".equals(topSort) && !"desc".equals(topSort)) {
                throw new DiscussException("topSort值有误,请传入asc或desc");
            }
        }
        Call<DiscussResponse<DiscussPageResult<DiscussCommentResult>>> call = discussApi.pageComment(appId, key, userId, topSort, sort, page, pageSize);
        try {
            Response<DiscussResponse<DiscussPageResult<DiscussCommentResult>>> res = call.execute();
            if (res.isSuccessful()) {
                if (res.body().getCode() == 1) {
                    return res.body().getData();
                }
                throw new DiscussException(res.body().getCode(), res.body().getMsg());
            }
            throw new DiscussException(50000, "评论服务访问异常[" + res.code() + "]");
        } catch (IOException e) {
            e.printStackTrace();
            throw new DiscussException(50000, "评论服务访问异常");
        }
    }

    /**
     * 搜索查询评论列表（可用于后台管理）
     * <p>
     * 异常错误码:
     * 40000: 参数异常
     * 50000: 评论服务访问异常
     *
     * @param searchMap 评论搜索请求map [必须]
     * @param page      第几页
     * @param pageSize  每页大小
     * @return
     */
    public DiscussPageResult<DiscussCommentSearchResult> searchComment(DiscussCommentSearchMap searchMap, Integer page, Integer pageSize, Integer top) throws DiscussException {
        String appId = discussProperties.getAppid();
        if (StringUtils.isEmpty(appId)) {
            throw new DiscussException("appId没有正确配置");
        }
        if (StringUtils.isEmpty(searchMap.getSort())) {
            searchMap.setSort("asc");
        } else {
            if (!"asc".equals(searchMap.getSort()) && !"desc".equals(searchMap.getSort())) {
                throw new DiscussException("sort值有误,请传入asc或desc");
            }
        }
        Long startTime = searchMap.getStartTime() != null ? searchMap.getStartTime().getTime() : null;
        Long endTime = searchMap.getEndTime() != null ? searchMap.getEndTime().getTime() : null;
        Call<DiscussResponse<DiscussPageResult<DiscussCommentSearchResult>>> call = discussApi.searchComment(appId, searchMap.getKey(), searchMap.getContent(), searchMap.getUserIds(), searchMap.getUserName(), searchMap.getStatus(), startTime, endTime, searchMap.getSort(), top, page, pageSize);
        try {
            Response<DiscussResponse<DiscussPageResult<DiscussCommentSearchResult>>> res = call.execute();
            if (res.isSuccessful()) {
                if (res.body().getCode() == 1) {
                    return res.body().getData();
                }
                throw new DiscussException(res.body().getCode(), res.body().getMsg());
            }
            throw new DiscussException(50000, "评论服务访问异常[" + res.code() + "]");
        } catch (IOException e) {
            e.printStackTrace();
            throw new DiscussException(50000, "评论服务访问异常");
        }
    }


    /**
     * 更新评论状态
     * <p>
     * 异常错误码:
     * 40000: 参数异常
     * 40001: 评论不存在
     * 40002: 权限不足
     * 50000: 评论服务访问异常
     *
     * @param commentId 评论id  [必须]
     * @param status    状态 0:被作废 1:正常 [必须]
     * @return
     */
    public void updateCommentStatus(String commentId, int status) throws DiscussException {
        String appId = discussProperties.getAppid();
        if (StringUtils.isEmpty(appId)) {
            throw new DiscussException("appId没有正确配置");
        }
        if (StringUtils.isEmpty(commentId)) {
            throw new DiscussException("评论id不能为空");
        }
        if (status != 0 && status != 1) {
            throw new DiscussException("状态值错误");
        }

        Call<DiscussResponse> call = discussApi.updateCommentStatus(appId, commentId, status);
        checkResponse(call);
    }

    /**
     * 置顶评论
     * <p>
     * 异常错误码:
     * 40000: 参数异常
     * 40001: 评论不存在
     * 40002: 权限不足
     * 50000: 评论服务访问异常
     *
     * @param commentId 评论id  [必须]
     * @return
     */
    public void topComment(String commentId) throws DiscussException {
        String appId = discussProperties.getAppid();
        if (StringUtils.isEmpty(appId)) {
            throw new DiscussException("appId没有正确配置");
        }
        if (StringUtils.isEmpty(commentId)) {
            throw new DiscussException("评论id不能为空");
        }
        Call<DiscussResponse> call = discussApi.topComment(appId, commentId);
        checkResponse(call);
    }

    /**
     * 取消置顶评论
     * <p>
     * 异常错误码:
     * 40000: 参数异常
     * 40001: 评论不存在
     * 40002: 权限不足
     * 50000: 评论服务访问异常
     *
     * @param commentId 评论id  [必须]
     * @return
     */
    public void cancelTopComment(String commentId) throws DiscussException {
        String appId = discussProperties.getAppid();
        if (StringUtils.isEmpty(appId)) {
            throw new DiscussException("appId没有正确配置");
        }
        if (StringUtils.isEmpty(commentId)) {
            throw new DiscussException("评论id不能为空");
        }
        Call<DiscussResponse> call = discussApi.cancelTopComment(appId, commentId);
        checkResponse(call);
    }


    /**
     * 删除评论
     * <p>
     * 异常错误码:
     * 40000: 参数异常
     * 40001: 评论不存在
     * 40002: 权限不足
     * 50000: 评论服务访问异常
     *
     * @param commentId 评论id [必须]
     * @return
     */
    public void deleteComment(String commentId) throws DiscussException {
        String appId = discussProperties.getAppid();
        if (StringUtils.isEmpty(appId)) {
            throw new DiscussException("appId没有正确配置");
        }
        if (StringUtils.isEmpty(commentId)) {
            throw new DiscussException("评论id不能为空");
        }

        Call<DiscussResponse> call = discussApi.deleteComment(appId, commentId);
        checkResponse(call);
    }


    /**
     * 获取评论数
     * <p>
     * 异常错误码:
     * 40000: 参数异常
     * 40001: 评论不存在
     * 40002: 权限不足
     * 50000: 评论服务访问异常
     *
     * @param key       评论归属key [非必须]
     * @param status    状态 0:被作废 1:正常 [非必须]
     * @param startTime 统计时间范围：开始时间 [非必须]
     * @param endTime   统计时间范围：结束时间 [非必须]
     * @return
     */
    public long getCommentCount(String key, Integer status, Date startTime, Date endTime) throws DiscussException {
        String appId = discussProperties.getAppid();
        if (StringUtils.isEmpty(appId)) {
            throw new RuntimeException("appId没有正确配置");
        }
        Long startTime_ = startTime != null ? startTime.getTime() : null;
        Long endTime_ = endTime != null ? endTime.getTime() : null;
        Call<DiscussResponse<Long>> call = discussApi.getCommentCount(appId, key, status, startTime_, endTime_);
        try {
            Response<DiscussResponse<Long>> res = call.execute();
            if (res.isSuccessful()) {
                if (res.body().getCode() == 1) {
                    return res.body().getData();
                }
                throw new DiscussException(res.body().getCode(), res.body().getMsg());
            }
            throw new DiscussException(50000, "评论服务访问异常[" + res.code() + "]");
        } catch (IOException e) {
            e.printStackTrace();
            throw new DiscussException(50000, "评论服务访问异常");
        }
    }

    /**
     * 获取多个key下的评论人数
     * <p>
     * 异常错误码:
     * 40000: 参数异常
     * 40001: 评论不存在
     * 40002: 权限不足
     * 50000: 评论服务访问异常
     *
     * @param keys 评论归属key数组 [必须]
     * @return
     */
    public List<DiscussCommentUserCountResult> getCommentUserCountByKeys(List<String> keys) throws DiscussException {
        String appId = discussProperties.getAppid();
        if (StringUtils.isEmpty(appId)) {
            throw new RuntimeException("appId没有正确配置");
        }
        Call<DiscussResponse<List<DiscussCommentUserCountResult>>> call = discussApi.getCommentUserCountByKeys(appId, keys);
        try {
            Response<DiscussResponse<List<DiscussCommentUserCountResult>>> res = call.execute();
            if (res.isSuccessful()) {
                if (res.body().getCode() == 1) {
                    return res.body().getData();
                }
                throw new DiscussException(res.body().getCode(), res.body().getMsg());
            }
            throw new DiscussException(50000, "评论服务访问异常[" + res.code() + "]");
        } catch (IOException e) {
            e.printStackTrace();
            throw new DiscussException(50000, "评论服务访问异常");
        }
    }

    private void checkResponse(Call<DiscussResponse> call) throws DiscussException {
        try {
            Response<DiscussResponse> res = call.execute();
            if (res.isSuccessful()) {
                if (res.body().getCode() != 1) {
                    throw new DiscussException(res.body().getCode(), res.body().getMsg());
                }
                return;
            }
            throw new DiscussException(50000, "评论服务访问异常[" + res.code() + "]");
        } catch (IOException e) {
            e.printStackTrace();
            throw new DiscussException(50000, "评论服务访问异常");
        }
    }

}
