package com.pcauto.lib.wx.ma.service;

import cn.binarywang.wx.miniapp.api.WxMaService;
import cn.binarywang.wx.miniapp.bean.WxMaJscode2SessionResult;
import com.pcauto.lib.core.mvc.error.CodedException;
import com.pcauto.lib.core.mvc.vo.GenericResponse;
import com.pcauto.lib.wx.ma.vo.WxMaLoginVo;
import me.chanjar.weixin.common.error.WxErrorException;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;

/**
 * <p>
 * 小程序授权service
 * </p>
 *
 * @author Sam
 * @since 2019-07-10
 */
@Service
public class WxMaAuthService {

    @Autowired
    private WxMaService wxMaService;

    @Autowired(required = false)
    private WxMaBaseService wxMaBaseService;

    /**
     * 获取小程序全局唯一后台接口调用凭据（access_token）。调调用绝大多数后台接口时都需使用 access_token，开发者需要进行妥善保存。
     * <p>
     * 文档: https://developers.weixin.qq.com/miniprogram/dev/api-backend/open-api/access-token/auth.getAccessToken.html
     *
     * @return
     */
    public String getAccessToken() {
        try {
            return wxMaService.getAccessToken();
        } catch (WxErrorException e) {
            throw new CodedException(e.getError().getErrorCode(), e.getError().getErrorMsg());
        }
    }

    /**
     * 登录获取
     *
     * @param code
     * @return
     */
    public GenericResponse<WxMaLoginVo> login(String code) {
        if (wxMaBaseService == null) {
            throw new RuntimeException("=============>>> 大哥,你忘记实现WxMaBaseService了 <<<=============");
        }
        WxMaLoginVo login = wxMaBaseService.login(code2Session(code));
        if (login == null || StringUtils.isEmpty(login.getToken())) {
            throw new CodedException(40002, "登录失败");
        }
        return new GenericResponse<>(1, "登录成功", login);
    }

    /**
     * 退出登录
     *
     * @return
     */
    public GenericResponse<String> logout() {
        if (wxMaBaseService == null) {
            throw new RuntimeException("=============>>> 大哥,你忘记实现WxMaBaseService了 <<<=============");
        }
        HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
        wxMaBaseService.logout(request);
        return new GenericResponse<>();
    }

    /**
     * 登录凭证校验。通过 wx.login 接口获得临时登录凭证 code 后传到开发者服务器调用此接口完成登录流程。更多使用方法详见 小程序登录。
     * <p>
     * 文档: https://developers.weixin.qq.com/miniprogram/dev/api-backend/open-api/login/auth.code2Session.html
     *
     * @param code
     * @return
     */
    public WxMaJscode2SessionResult code2Session(String code) {
        if (StringUtils.isBlank(code)) {
            throw new CodedException(40001, "code 不能为空");
        }
        try {
            return wxMaService.getUserService().getSessionInfo(code);
        } catch (WxErrorException e) {
            throw new CodedException(e.getError().getErrorCode(), e.getError().getErrorMsg());
        }
    }

}
